<?php
if (!defined('ABSPATH')) {
    exit('No direct access allowed.');
}

class Webafra_Store_Brand_Filter_Widget extends WP_Widget
{
    function __construct()
    {
        parent::__construct('webafra_store_brand_filter', __('Theme: Brand filter', 'webafra-store'), [
            'description' => __('Product brand filter', 'webafra-store')
        ]);
    }

    public function widget($args, $instance)
    {
        if (is_shop() || is_product_taxonomy()) {
            $current_products = $this->current_products_query();

            if (!empty($current_products) && !is_tax(get_theme_mod('shop-brand-taxonomy', 'pa_brand')) && !is_product()) {
                $result_brands = $this->get_products_brands($current_products);
                $title = apply_filters('widget_title', $instance['title']);
                $show_brand_logo = isset($instance['show_brand_logo']) ? $instance['show_brand_logo'] : '';

                echo $args['before_widget'];

                if (!empty($title)) {
                    echo $args['before_title'] . $title . $args['after_title'];
                }

                $product_brand = isset($_GET['product_brand']) ? explode(',', $_GET['product_brand']) : [];

                if (! empty($result_brands) && ! is_wp_error($result_brands)) { ?>
                    <ul class="brand-filter">
                        <?php
                        foreach ($result_brands as $brand) {
                            $brand = get_term($brand); ?>
                            <li class="filter-item<?php echo in_array($brand->term_id, $product_brand) ? ' is-active' : ''; ?>">
                                <a rel="nofollow" title="<?= $brand->name; ?>" href="<?php echo esc_attr($this->get_link($brand->term_id)); ?>">
                                    <?php if (!empty($show_brand_logo)) {
                                        $brand_thumbnail_src = wc_placeholder_img_src();
                                        if ($brand_thumbnail_id = get_term_meta($brand->term_id, 'product_attribute_image', true)) {
                                            if ($brand_thumbnail_image = wp_get_attachment_image_src($brand_thumbnail_id, 'woocommerce_thumbnail')) {
                                                $brand_thumbnail_src = $brand_thumbnail_image[0];
                                            }
                                        }
                                        echo '<img ' . ws_image_lazy_load_attr() . ' src="' . $brand_thumbnail_src . '" alt="' . $brand->name . '" class="filter-brand-logo">';
                                    }
                                    echo $brand->name;
                                    ?>
                                </a>
                            </li>
                        <?php } ?>
                    </ul>
                <?php } else {
                    _e('No brand found', 'webafra-store');
                } ?>
        <?php echo $args['after_widget'];
            }
        }
    }

    public function form($instance)
    {
        $title = isset($instance['title']) ? esc_attr($instance['title']) : __('Brand filter', 'webafra-store');
        $show_brand_logo = isset($instance['show_brand_logo']) ? $instance['show_brand_logo'] : ''; ?>
        <p>
            <label for="<?php echo $this->get_field_id('title'); ?>"><?php _e('Title', 'webafra-store'); ?></label>
            <input id="<?php echo $this->get_field_id('title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo esc_attr($title); ?>" />
        </p>
        <p>
            <label for="<?php echo $this->get_field_id('show_brand_logo'); ?>">
                <input type="checkbox" id="<?php echo $this->get_field_id('show_brand_logo'); ?>" value="true" name="<?php echo $this->get_field_name('show_brand_logo'); ?>" <?php checked($show_brand_logo, 'true'); ?> />
                <?php _e('Show brand logo', 'webafra-store') ?>
            </label>
        </p>
<?php
    }

    public function update($new_instance, $old_instance)
    {
        $instance = [];

        $instance['title'] = (! empty($new_instance['title'])) ? strip_tags($new_instance['title']) : '';
        $instance['show_brand_logo'] = sanitize_text_field($new_instance['show_brand_logo']);

        return $instance;
    }

    private function get_link($brand_id)
    {
        $base_link = ws_wc_shop_page_link(true);
        $link = remove_query_arg('product_brand', $base_link);
        $current_product_brand = isset($_GET['product_brand']) ? explode(',', $_GET['product_brand']) : [];
        $option_is_set = in_array($brand_id, $current_product_brand);

        if (!in_array($brand_id, $current_product_brand)) {
            $current_product_brand[] = $brand_id;
        }

        foreach ($current_product_brand as $key => $value) {
            if ($option_is_set && $value == $brand_id) {
                unset($current_product_brand[$key]);
            }
        }

        if ($current_product_brand) {
            asort($current_product_brand);
            $link = add_query_arg('product_brand', implode(',', $current_product_brand), $link);
        }

        $link = str_replace('%2C', ',', $link);
        return $link;
    }

    private function current_products_query()
    {
        $args = [
            'posts_per_page' => -1,
            'post_type' => 'product',
            'fields' => 'ids',
            'tax_query' => [
                [
                    'taxonomy' => get_theme_mod('shop-brand-taxonomy', 'pa_brand'),
                    'operator' => 'EXISTS',
                ]
            ]
        ];

        $cat = get_queried_object();

        if (is_a($cat, 'WP_Term')) {
            $cat_id = $cat->term_id;
            $cat_id_array = get_term_children($cat_id, $cat->taxonomy);
            $cat_id_array[] = $cat_id;
            $args['tax_query'][] = [
                'taxonomy' => $cat->taxonomy,
                'field' => 'term_id',
                'terms' => $cat_id_array,
            ];
        }

        if (get_option('woocommerce_hide_out_of_stock_items') === 'yes') {
            $args['meta_query'] = [
                [
                    'key' => '_stock_status',
                    'value' => 'outofstock',
                    'compare' => 'NOT IN',
                ]
            ];
        }

        $wp_query = new WP_Query($args);

        wp_reset_postdata();

        return $wp_query->posts;
    }

    private function get_products_brands($product_ids)
    {
        $brand_taxonomy = get_theme_mod('shop-brand-taxonomy', 'pa_brand');
        $product_ids = implode(',', array_map('intval', $product_ids));

        global $wpdb;

        $brand_ids = $wpdb->get_col(
            "SELECT DISTINCT t.term_id
            FROM {$wpdb->prefix}terms AS t
            INNER JOIN {$wpdb->prefix}term_taxonomy AS tt
            ON t.term_id = tt.term_id
            INNER JOIN {$wpdb->prefix}term_relationships AS tr
            ON tr.term_taxonomy_id = tt.term_taxonomy_id
            WHERE tt.taxonomy = '$brand_taxonomy'
            AND tr.object_id IN ( $product_ids)"
        );

        return $brand_ids ? $brand_ids : false;
    }
}

add_action('widgets_init', function () {
    register_widget('Webafra_Store_Brand_Filter_Widget');
});
