<?php
if (!defined('ABSPATH')) {
    exit('No direct access allowed.');
}

class Webafra_Store_Theme_Settings
{
    public static $ins = null;

    public $post_types = [
        'post',
        'page',
    ];

    public static function ins()
    {
        if (is_null(self::$ins)) {
            self::$ins = new self();
        }

        return self::$ins;
    }

    public function __construct()
    {
        if (class_exists('WooCommerce')) {
            $this->post_types[] = 'product';
        }

        if (function_exists('aiovg_fs')) {
            $this->post_types[] = 'aiovg_videos';
        }

        add_action('add_meta_boxes', [$this, 'meta_box']);

        add_action('save_post', [$this, 'save']);

        add_action('wp_head', [$this, 'frontend'], 99, 0);

        add_action('init', [$this, 'taxonomies']);
    }

    public function frontend(): void
    {
        if (is_single() || is_page()) {
            global $post;

            $post_schema = get_post_meta($post->ID, '_ws_post_schema', true);

            if (!empty($post_schema)) {
                echo '<script type="application/ld+json">' . $post_schema . '</script>';
            }

            $post_meta_keywords = get_post_meta($post->ID, '_ws_post_meta_keywords', true);

            if (!empty($post_meta_keywords)) {
                echo '<meta name="keywords" content="' . $post_meta_keywords . '">';
            }
        }

        if (is_archive()) {
            $queried_object_id = get_queried_object_id();
            $term_meta_keywords = get_term_meta($queried_object_id, '_ws_term_meta_keywords', true);

            if (!empty($term_meta_keywords)) {
                echo '<meta name="keywords" content="' . $term_meta_keywords . '">';
            }
        }
    }

    public function callback($post): void
    {
        wp_nonce_field('ws_theme_settings', 'ws_theme_settings_nonce');

        get_template_part('templates/admin/settings-meta-post', null, [
            'post' => $post,
        ]);
    }

    public function meta_box(): void
    {
        foreach ($this->post_types as $post_type) {
            add_meta_box(
                'ws-theme-settings',
                __('Theme settings', 'webafra-store'),
                [$this, 'callback'],
                $post_type,
                'advanced',
                'high'
            );
        }
    }

    public function save(int $post_id): void
    {
        if (!isset($_POST['ws_theme_settings_nonce'])) {
            return;
        }

        if (!wp_verify_nonce($_POST['ws_theme_settings_nonce'], 'ws_theme_settings')) {
            return;
        }

        if (wp_is_post_autosave($post_id)) {
            return;
        }

        if (!current_user_can('manage_options', $post_id)) {
            return;
        }

        if (!current_user_can('manage_options', $post_id)) { // edit_page
            return;
        }

        update_post_meta($post_id, '_ws_theme_settings_hidden_title', isset($_POST['ws-theme-settings-hidden-title']) ? 1 : 0);

        update_post_meta($post_id, '_ws_theme_settings_disable_sidebar', isset($_POST['ws-theme-settings-disable-sidebar']) ? 1 : 0);

        update_post_meta($post_id, '_ws_post_schema', sanitize_textarea_field($_POST['ws-post-schema-form']));

        update_post_meta($post_id, '_ws_post_meta_keywords', sanitize_textarea_field($_POST['ws-post-meta-keywords']));

        if (get_post_type($post_id) == 'product') {
            $product_offer_finishing_at = sanitize_text_field($_POST['ws-product-offer-finishing-at']);

            if (!empty($product_offer_finishing_at)) {
                $product_offer_finishing_at_jalali = explode('-', $product_offer_finishing_at);

                if (count($product_offer_finishing_at_jalali) == 3) {
                    $product_offer_finishing_at_gregorian = jalali_to_gregorian($product_offer_finishing_at_jalali[0], $product_offer_finishing_at_jalali[1], $product_offer_finishing_at_jalali[2], '-');
                    $product_offer_finishing_at = strtotime($product_offer_finishing_at_gregorian);
                } else {
                    $product_offer_finishing_at_jalali = '';
                }
            }

            update_post_meta($post_id, '_woo_product_offerslider_timer', $product_offer_finishing_at);
        }
    }

    public function taxonomies()
    {
        $object_taxonomies = get_object_taxonomies($this->post_types);

        if (!empty($object_taxonomies)) {
            foreach ($object_taxonomies as $taxonomy) {
                add_action($taxonomy . '_edit_form_fields', [$this, 'taxonomy_edit_form_fields']);
                add_action($taxonomy . '_add_form_fields', [$this, 'taxonomy_edit_form_fields']);
                add_action('edited_' . $taxonomy, [$this, 'taxonomy_save_form_fields'], 10, 1);
                add_action('created_' . $taxonomy, [$this, 'taxonomy_save_form_fields'], 10, 1);
            }
        }
    }

    public function taxonomy_edit_form_fields($term)
    {
        echo '<tr class="form-field">';
        echo '<th valign="top" scope="row">';
        echo '<label for="meta-keywords">' . __('Meta keywords', 'webafra-store') . '</label>';
        echo '</th>';
        echo '<td>';
        echo '<textarea style="min-height:100px;width:100%;" id="meta-keywords" name="meta-keywords">' . get_term_meta($term->term_id, '_ws_term_meta_keywords', true) . '</textarea>';
        echo '</td>';
        echo '</tr>';
    }

    public function taxonomy_save_form_fields($term_id)
    {
        if (isset($_POST['meta-keywords'])) {
            update_term_meta($term_id, '_ws_term_meta_keywords', sanitize_textarea_field($_POST['meta-keywords']));
        }
    }
}

Webafra_Store_Theme_Settings::ins();
