<?php

use Elementor\Controls_Manager;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Utils;
use Elementor\Widget_Base;

if (!defined('ABSPATH')) {
    exit('No direct access allowed.');
}

class Webafra_Store_Elementor_Widget_Products_Carousel extends Widget_Base
{
    public function get_name()
    {
        return 'webafra_store_elementor_products_carousel';
    }

    public function get_title()
    {
        return esc_html__('Products (Carousel)', 'webafra-store');
    }

    public function get_icon()
    {
        return 'eicon-products';
    }

    public function get_script_depends()
    {
        return [
            'webafra-store-owl.carousel.min',
            'webafra-store-products',
        ];
    }

    public function get_style_depends()
    {
        return [
            'webafra-store-owl.carousel.min',
            'webafra-store-owl.theme.default.min',
            'webafra-store-products',
        ];
    }

    public function get_categories()
    {
        return ['webafra-store-widgets'];
    }

    protected function register_controls()
    {
        $this->start_controls_section('query', [
            'label' => esc_html__('Query', 'webafra-store'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $this->add_control('source_type', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Source', 'webafra-store'),
            'options' => [
                'latest' => esc_html__('Latest', 'webafra-store'),
                'featured' => esc_html__('Featured', 'webafra-store'),
                'sale' => esc_html__('Sale', 'webafra-store'),
                'sale_dates' => esc_html__('Sale with dates', 'webafra-store'),
            ],
            'default' => ['latest'],
        ]);

        $categories = new WP_Term_Query([
            'taxonomy' => 'product_cat',
            'hide_empty' => 0,
            'number' => 0,
        ]);

        $categories_new = [];

        if (count($categories->get_terms())) {
            foreach (array_values($categories->get_terms()) as $cat) {
                $categories_new[$cat->term_id] = $cat->name;
            }
        }

        $this->add_control('include_category', [
            'label' => esc_html__('Include', 'webafra-store'),
            'type' => Controls_Manager::SELECT2,
            'label_block' => true,
            'multiple' => true,
            'options' => $categories_new,
            'default' => [],
        ]);

        $this->add_control('limit', [
            'label' => esc_html__('Limit', 'webafra-store'),
            'type' => Controls_Manager::NUMBER,
            'default' => 8,
        ]);

        $this->end_controls_section();

        $this->start_controls_section('settings', [
            'label' => esc_html__('Settings', 'webafra-store'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $this->add_control('title_type', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Title type', 'webafra-store'),
            'options' => [
                'custom' => esc_html__('Custom', 'webafra-store'),
                'term' => esc_html__('Term', 'webafra-store'),
            ],
            'default' => 'custom',
        ]);

        $this->add_control('title', [
            'label' => esc_html__('Title', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => esc_html__('Title', 'webafra-store'),
            'condition' => [
                'title_type' => 'custom'
            ]
        ]);

        $this->add_control('title_link', [
            'label' => esc_html__('Title link', 'webafra-store'),
            'type' => Controls_Manager::URL,
            'options' => [
                'url',
                'is_external',
                'nofollow'
            ],
            'default' => [
                'url' => '',
                'is_external' => false,
                'nofollow' => true,
            ],
            'label_block' => true,
            'condition' => [
                'title_type' => 'custom'
            ]
        ]);

        $this->add_control('title_by_term', [
            'label' => esc_html__('Select term for title', 'webafra-store'),
            'type' => Controls_Manager::SELECT2,
            'label_block' => true,
            'multiple' => false,
            'options' => $categories_new,
            'default' => [],
            'condition' => [
                'title_type' => 'term'
            ]
        ]);

        $this->add_control('title_tag', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Title tag', 'webafra-store'),
            'options' => [
                'h1' => esc_html__('H1', 'webafra-store'),
                'h2' => esc_html__('H2', 'webafra-store'),
                'h3' => esc_html__('H3', 'webafra-store'),
                'h4' => esc_html__('H4', 'webafra-store'),
            ],
            'default' => 'h2',
            'separator' => 'after',
        ]);

        $this->add_control('thumbnail_size', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Thumbnail size', 'webafra-store'),
            'multiple' => false,
            'options' => ws_get_thumbnail_sizes(),
            'default' => 'woocommerce_thumbnail',
        ]);

        $this->add_control('thumbnail_size_custom_width', [
            'label' => esc_html__('Width', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => esc_html__('Width', 'webafra-store'),
            'condition' => [
                'thumbnail_size' => 'custom'
            ]
        ]);

        $this->add_control('thumbnail_size_custom_height', [
            'label' => esc_html__('Height', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => esc_html__('Height', 'webafra-store'),
            'condition' => [
                'thumbnail_size' => 'custom'
            ]
        ]);

        $this->add_control('theme', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Theme', 'webafra-store'),
            'options' => [
                'one' => esc_html__('One', 'webafra-store'),
                'two' => esc_html__('Two', 'webafra-store'),
            ],
            'default' => 'one',
        ]);

        $this->add_control('show_rating', [
            'label' => esc_html__('Show rating', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
            'condition' => [
                'theme' => 'two'
            ]
        ]);

        $this->add_control('show_cart_button', [
            'label' => esc_html__('Show cart button', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('show_onsale_badge', [
            'label' => esc_html__('Show onsale badge', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('enable_section', [
            'label' => esc_html__('Enable section', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->end_controls_section();

        $this->start_controls_section('carousel', [
            'label' => esc_html__('Carousel', 'webafra-store'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $this->add_responsive_control('carousel_limit', [
            'label' => esc_html__('Carousel limit', 'webafra-store'),
            'type' => Controls_Manager::NUMBER,
            'default' => 4,
        ]);

        $this->add_control('carousel_gap', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Gap', 'webafra-store'),
            'size_units' => ['px'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 50,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => 15,
            ],
        ]);

        $this->add_control('autoplay', [
            'label' => esc_html__('Autoplay', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'no',
        ]);

        $this->add_control('show_slide_progress', [
            'label' => esc_html__('Show slide progress', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'no',
            'condition' => [
                'autoplay' => 'yes'
            ]
        ]);

        $this->add_control('show_nav', [
            'label' => esc_html__('Show nav', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('nav_location', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Nav location', 'webafra-store'),
            'options' => [
                'outside' => esc_html__('Outside', 'webafra-store'),
                'inside' => esc_html__('Inside', 'webafra-store'),
            ],
            'default' => 'outside',
            'condition' => [
                'show_nav' => 'yes'
            ]
        ]);

        $this->add_control('auto_width', [
            'label' => esc_html__('Auto width', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'no',
        ]);

        $this->add_control('stage_padding', [
            'label' => esc_html__('Stage padding', 'webafra-store'),
            'type' => Controls_Manager::NUMBER,
            'default' => 0,
        ]);

        $this->end_controls_section();

        $this->start_controls_section('additional_item', [
            'label' => esc_html__('Additional item', 'webafra-store'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $this->add_control('additional_item_enable', [
            'label' => esc_html__('Enable', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'no',
        ]);

        $this->add_control('additional_item_image', [
            'label' => esc_html__('Image', 'webafra-store'),
            'type' => Controls_Manager::MEDIA,
            'default' => [
                'url' => Utils::get_placeholder_image_src(),
            ],
            'condition' => [
                'additional_item_enable' => 'yes'
            ]
        ]);

        $this->add_control('additional_item_button_text', [
            'label' => esc_html__('Button text', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
            'condition' => [
                'additional_item_enable' => 'yes'
            ]
        ]);

        $this->add_control('additional_item_link', [
            'label' => esc_html__('Link', 'webafra-store'),
            'type' => Controls_Manager::URL,
            'options' => [
                'url',
                'is_external',
                'nofollow'
            ],
            'default' => [
                'url' => '',
                'is_external' => false,
                'nofollow' => true,
            ],
            'label_block' => true,
            'condition' => [
                'additional_item_enable' => 'yes'
            ]
        ]);

        $this->end_controls_section();

        $this->start_controls_section('additional_item_end', [
            'label' => esc_html__('Additional item (End)', 'webafra-store'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $this->add_control('additional_item_end_enable', [
            'label' => esc_html__('Enable', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'no',
        ]);

        $this->add_control('additional_item_end_label', [
            'label' => esc_html__('Label', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
            'condition' => [
                'additional_item_end_enable' => 'yes'
            ]
        ]);

        $this->add_control('additional_item_end_link', [
            'label' => esc_html__('Link', 'webafra-store'),
            'type' => Controls_Manager::URL,
            'options' => [
                'url',
                'is_external',
                'nofollow'
            ],
            'default' => [
                'url' => '',
                'is_external' => false,
                'nofollow' => true,
            ],
            'label_block' => true,
            'condition' => [
                'additional_item_end_enable' => 'yes'
            ]
        ]);

        $this->end_controls_section();

        $this->start_controls_section('section_style', [
            'label' => esc_html__('Section', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
        ]);

        $this->add_responsive_control('section_title', [
            'label' => esc_html__('Title alignment', 'webafra-store'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => esc_html__('Left', 'webafra-store'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => esc_html__('Center', 'webafra-store'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => esc_html__('Right', 'webafra-store'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'right',
            'toggle' => true,
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-head .head-title' => 'text-align: {{VALUE}};',
            ],
        ]);

        $this->add_control('section_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products.section-box' => 'background-color: {{VALUE}}',
            ],
            'condition' => [
                'enable_section' => 'yes'
            ]
        ]);

        $this->add_responsive_control('height', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Height', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products' => 'height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('section_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_group_control(Group_Control_Box_Shadow::get_type(), [
            'name' => 'section_shadow',
            'selector' => '{{WRAPPER}} .ws-products',
        ]);

        $this->end_controls_section();

        $this->start_controls_section('product_style', [
            'label' => esc_html__('Product', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
        ]);

        $this->add_responsive_control('product_thumbnail_width_size', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Thumbnail width', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail img' => 'width: {{SIZE}}{{UNIT}};'
            ],
        ]);

        $this->add_responsive_control('product_thumbnail_height_size', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Thumbnail height', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail img' => 'height: {{SIZE}}{{UNIT}};'
            ],
        ]);

        $this->add_responsive_control('product_thumbnail_border', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Thumbnail border', 'webafra-store'),
            'options' => [
                'none' => esc_html__('None', 'webafra-store'),
                'solid' => esc_html__('Solid', 'webafra-store'),
                'dashed' => esc_html__('Dashed', 'webafra-store')
            ],
            'default' => 'none',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail img' => 'border-style: {{VALUE}};',
            ]
        ]);

        $this->add_control('product_thumbnail_border_color', [
            'label' => esc_html__('Thumbnail border color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail img' => 'border-color: {{VALUE}}',
            ],
            'condition' => [
                'product_thumbnail_border!' => 'none'
            ]
        ]);

        $this->add_responsive_control('product_thumbnail_border_width', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Thumbnail border width', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail img' => 'border-top-width: {{TOP}}{{UNIT}}; border-right-width: {{RIGHT}}{{UNIT}}; border-left-width: {{LEFT}}{{UNIT}}; border-bottom-width: {{BOTTOM}}{{UNIT}};'
            ],
            'condition' => [
                'product_thumbnail_border!' => 'none'
            ]
        ]);

        $this->add_responsive_control('product_thumbnail_border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Thumbnail border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('product_thumbnail_margin', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Thumbnail margin', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('product_title_alignment', [
            'label' => esc_html__('Title alignment', 'webafra-store'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => esc_html__('Left', 'webafra-store'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => esc_html__('Center', 'webafra-store'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => esc_html__('Right', 'webafra-store'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'right',
            'toggle' => true,
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-title' => 'text-align: {{VALUE}};',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Title', 'webafra-store'),
            'name' => 'product_title_typography',
            'selector' => '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-title a'
        ]);

        $this->add_control('product_title_color', [
            'label' => esc_html__('Title color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-title a' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('product_price_heading', [
            'type' => Controls_Manager::HEADING,
            'label' => esc_html__('Price', 'webafra-store'),
        ]);

        $this->add_responsive_control('product_price_alignment', [
            'label' => esc_html__('Price alignment', 'webafra-store'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'flex-end' => [
                    'title' => esc_html__('Left', 'webafra-store'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => esc_html__('Center', 'webafra-store'),
                    'icon' => 'eicon-text-align-center',
                ],
                'flex-start' => [
                    'title' => esc_html__('Right', 'webafra-store'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'right',
            'toggle' => true,
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-price .price' => 'align-items: {{VALUE}};',
                '{{WRAPPER}} .ws-products .products-wrapper.theme-two .product-loop.product-type-variable .product-details .product-price .price' => 'justify-content: {{VALUE}};',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Price', 'webafra-store'),
            'name' => 'product_price_typography',
            'selector' => '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-price .price'
        ]);

        $this->add_control('product_price_color', [
            'label' => esc_html__('Price color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-price .price' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Price del', 'webafra-store'),
            'name' => 'product_price_del_typography',
            'selector' => '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-price .price del'
        ]);

        $this->add_control('product_price_del_color', [
            'label' => esc_html__('Price del color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-price .price del' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('product_price_height', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Height', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-price' => 'height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('product_rating_heading', [
            'type' => Controls_Manager::HEADING,
            'label' => esc_html__('Rating', 'webafra-store'),
            'condition' => [
                'show_rating' => 'yes',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Rating typography', 'webafra-store'),
            'name' => 'product_rating_number_typography',
            'selector' => '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-footer .product-rating span',
            'condition' => [
                'show_rating' => 'yes',
            ],
        ]);

        $this->add_control('product_rating_number_color', [
            'label' => esc_html__('Rating color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'condition' => [
                'show_rating' => 'yes',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-footer .product-rating span' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('product_rating_icon_size', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Rating icon size', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'condition' => [
                'show_rating' => 'yes',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-details .product-footer .product-rating svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_control('product_onsale_badge_hr', [
            'type' => Controls_Manager::DIVIDER,
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_onsale_badge_heading', [
            'label' => esc_html__('Onsale badge', 'webafra-store'),
            'type' => Controls_Manager::HEADING,
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Typography', 'webafra-store'),
            'name' => 'product_onsale_badge_typography',
            'selector' => '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail span.onsale',
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_onsale_badge_color', [
            'label' => esc_html__('Color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail span.onsale' => 'color: {{VALUE}};',
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_onsale_badge_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail span.onsale' => 'background-color: {{VALUE}};',
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_responsive_control('product_onsale_badge_border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail span.onsale' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_responsive_control('product_onsale_badge_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop .product-thumbnail span.onsale' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop' => 'background-color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('product_border', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Border', 'webafra-store'),
            'options' => [
                'none' => esc_html__('None', 'webafra-store'),
                'solid' => esc_html__('Solid', 'webafra-store'),
                'dashed' => esc_html__('Dashed', 'webafra-store')
            ],
            'default' => 'none',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop' => 'border-style: {{VALUE}};',
            ]
        ]);

        $this->add_control('product_border_color', [
            'label' => esc_html__('Border color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop' => 'border-color: {{VALUE}}',
            ],
            'condition' => ['product_border!' => 'none']
        ]);

        $this->add_responsive_control('product_border_width', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border width', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop' => 'border-top-width: {{TOP}}{{UNIT}}; border-right-width: {{RIGHT}}{{UNIT}}; border-left-width: {{LEFT}}{{UNIT}}; border-bottom-width: {{BOTTOM}}{{UNIT}};'
            ],
            'separator' => 'after',
            'condition' => ['product_border!' => 'none']
        ]);

        $this->add_responsive_control('product_border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('product_width', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Width', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop' => 'width: {{SIZE}}{{UNIT}};'
            ],
        ]);

        $this->add_responsive_control('product_margin', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Margin', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('product_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .product-loop' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_group_control(Group_Control_Box_Shadow::get_type(), [
            'name' => 'product_shadow',
            'selector' => '{{WRAPPER}} .ws-products .products-wrapper .product-loop',
        ]);

        $this->end_controls_section();

        $this->start_controls_section('additional_item_style', [
            'label' => esc_html__('Additional item', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
            'condition' => [
                'additional_item_enable' => 'yes'
            ]
        ]);

        $this->add_responsive_control('additional_item_image_width', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Image width', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item .additional-item-link .additional-item-image img' => 'width: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('additional_item_image_height', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Image height', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item .additional-item-link .additional-item-image img' => 'height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_control('additional_item_button_color', [
            'label' => esc_html__('Button color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item .additional-item-link .additional-item-button' => 'color: {{VALUE}}',
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item .additional-item-link .additional-item-button svg' => 'stroke: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('additional_item_button_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Button padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item .additional-item-link .additional-item-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_control('additional_item_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item' => 'background-color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('additional_item_width', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Width', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item' => 'width: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('additional_item_height', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Height', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item' => 'height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('additional_item_border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('additional_item_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('additional_item_margin', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Margin', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_group_control(Group_Control_Box_Shadow::get_type(), [
            'name' => 'additional_item_shadow',
            'selector' => '{{WRAPPER}} .ws-products .products-wrapper .additional-item',
        ]);

        $this->end_controls_section();

        $this->start_controls_section('additional_item_end_style', [
            'label' => esc_html__('Additional item (End)', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
            'condition' => [
                'additional_item_end_enable' => 'yes'
            ]
        ]);

        $this->add_control('additional_item_end_label_color', [
            'label' => esc_html__('Label color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item.end .additional-item-link .additional-item-label span' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Label typography', 'webafra-store'),
            'name' => 'additional_item_end_label_typography',
            'selector' => '{{WRAPPER}} .ws-products .products-wrapper .additional-item.end .additional-item-link .additional-item-label span'
        ]);

        $this->add_control('additional_item_end_icon_color', [
            'label' => esc_html__('Icon color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item.end .additional-item-link .more-icon' => 'border:1px solid {{VALUE}};color:{{VALUE}};',
            ],
        ]);

        $this->add_control('additional_item_end_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item.end' => 'background-color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('additional_item_end_width', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Width', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item.end' => 'width: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('additional_item_end_height', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Height', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item.end' => 'height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('additional_item_end_border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item.end' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('additional_item_end_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item.end' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('additional_item_end_margin', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Margin', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products .products-wrapper .additional-item.end' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_group_control(Group_Control_Box_Shadow::get_type(), [
            'name' => 'additional_item_end_shadow',
            'selector' => '{{WRAPPER}} .ws-products .products-wrapper .additional-item.end',
        ]);

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        $args = [
            'post_type' => 'product',
            'post_status' => 'publish',
            'ignore_sticky_posts' => 1,
            'posts_per_page' => !empty($settings['limit']) ? $settings['limit'] : 8,
            'orderby' => 'date',
            'order' => 'desc',
        ];

        if ($settings['source_type'] == 'featured') {
            $args['post__in'] = wc_get_featured_product_ids();
        } elseif ($settings['source_type'] == 'sale') {
            $args['post__in'] = wc_get_product_ids_on_sale();
        } elseif ($settings['source_type'] == 'sale_dates') {
            $args['post__in'] = wc_get_product_ids_on_sale();
            $args['meta_query'] = [
                'relation' => 'AND',
                [
                    'key' => '_sale_price_dates_to',
                    'value' => current_time('timestamp'),
                    'compare' => '>=',
                    'type' => 'numeric'
                ],
            ];
            $args['meta_key'] = '_sale_price_dates_from';
            $args['orderby'] = 'meta_value_num';
        }

        if (!empty($settings['include_category'])) {
            $args['tax_query'] = [
                [
                    'taxonomy' => 'product_cat',
                    'field' => 'term_id',
                    'terms' => $settings['include_category'],
                    'operator' => 'IN' // 'IN', 'NOT IN', 'AND'
                ]
            ];
        }

        $products = new WP_Query($args); ?>

        <?php if (count($products->get_posts())) : ?>
            <?php $has_sale_price_dates_to = 0; ?>
            <?php foreach ($products->get_posts() as $post) : ?>
                <?php $sale_price_dates_to = get_post_meta($post->ID, get_theme_mod('shop-product-offer-end-date-key', '_woo_product_offerslider_timer'), true); ?>
                <?php if (!empty($sale_price_dates_to)): ?>
                    <?php $has_sale_price_dates_to = 1; ?>
                <?php endif; ?>
            <?php endforeach; ?>

            <div class="ws-products<?= $settings['enable_section'] == 'yes' ? ' section-box' : ''; ?>">
                <?php if (!empty($settings['title']) || ($settings['show_nav'] == 'yes' && $settings['nav_location'] == 'outside')): ?>
                    <div class="products-head">
                        <div class="head-title">
                            <?php if ($settings['title_type'] == 'custom' && !empty($settings['title'])): ?>
                                <<?= $settings['title_tag']; ?>>
                                    <?php if (!empty($settings['title_link']['url'])): ?>
                                        <a href="<?= $settings['title_link']['url']; ?>" title="<?= $settings['title']; ?>" target="<?= $settings['title_link']['is_external'] == true ? '_blank' : '_self'; ?>"><?= $settings['title']; ?></a>
                                    <?php else: ?>
                                        <?= $settings['title']; ?>
                                    <?php endif; ?>
                                </<?= $settings['title_tag']; ?>>
                            <?php elseif ($settings['title_type'] == 'term' && !empty($settings['title_by_term'])): $title_by_term = get_term($settings['title_by_term']); ?>
                                <<?= $settings['title_tag']; ?>>
                                    <a href="<?= get_term_link($title_by_term); ?>" title="<?= $title_by_term->name; ?>"><?= $title_by_term->name; ?></a>
                                </<?= $settings['title_tag']; ?>>
                            <?php endif; ?>
                        </div>
                        <?php if ($settings['show_nav'] == 'yes' && $settings['nav_location'] == 'outside'): ?>
                            <div id="custom-nav-<?= $this->get_id(); ?>" class="custom-nav <?= $settings['nav_location']; ?>"></div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <?php if ($settings['show_slide_progress'] == 'yes'): ?>
                    <div class="owl-carousel-slide-progress" id="slide-progress-<?= $this->get_id(); ?>"></div>
                <?php endif; ?>

                <?php if ($settings['show_nav'] == 'yes' && $settings['nav_location'] == 'inside'): ?>
                    <div id="custom-nav-<?= $this->get_id(); ?>" class="custom-nav <?= $settings['nav_location']; ?>"></div>
                <?php endif; ?>

                <div id="webafra_store_elementor_products_carousel-<?= $this->get_id(); ?>" class="products-wrapper theme-<?= $settings['theme']; ?> owl-carousel owl-theme">
                    <?php if ($settings['additional_item_enable'] == 'yes'): ?>
                        <div class="additional-item">
                            <<?= !empty($settings['additional_item_link']['url']) ? 'a target="' . ($settings['additional_item_link']['is_external'] == true ? '_blank' : '_self') . '" href="' . esc_url($settings['additional_item_link']['url']) . '"' : 'div'; ?> class="additional-item-link">
                                <?php if (!empty($settings['additional_item_image']) && is_array($settings['additional_item_image'])) : ?>
                                    <?php if ($additional_item_image = wp_get_attachment_image_src($settings['additional_item_image']['id'], 'medium_large')): ?>
                                        <div class="additional-item-image">
                                            <img <?= ws_image_lazy_load_attr(); ?> alt="<?= get_post_meta($settings['additional_item_image']['id'], '_wp_attachment_image_alt', true); ?>" src="<?= $additional_item_image[0]; ?>" />
                                        </div>
                                    <?php endif; ?>
                                <?php endif; ?>

                                <?php if (!empty($settings['additional_item_button_text'])): ?>
                                    <div class="additional-item-button">
                                        <span><?= $settings['additional_item_button_text']; ?></span>
                                        <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">
                                            <path d="M15 18l-6-6 6-6" />
                                        </svg>
                                    </div>
                                <?php endif; ?>
                            </<?= !empty($settings['additional_item_link']['url']) ? 'a' : 'div'; ?>>
                        </div>
                    <?php endif; ?>

                    <?php foreach ($products->get_posts() as $post) : $product = wc_get_product($post); ?>
                        <div class="product-loop product-type-<?= $product->get_type(); ?>">
                            <?php
                            $thumbnail_size = $settings['thumbnail_size'];

                            if ($settings['thumbnail_size'] == 'custom' && (!empty($settings['thumbnail_size_custom_width']) && !empty($settings['thumbnail_size_custom_height']))) {
                                $thumbnail_size = [
                                    $settings['thumbnail_size_custom_width'],
                                    $settings['thumbnail_size_custom_height']
                                ];
                            }

                            $thumbnail_src = [
                                wc_placeholder_img_src($thumbnail_size),
                            ];

                            if ($thumbnail_id = get_post_thumbnail_id($product->get_id())) {
                                if ($thumbnail_image = wp_get_attachment_image_src($thumbnail_id, $thumbnail_size)) {
                                    $thumbnail_src = [
                                        $thumbnail_image[0]
                                    ];
                                }
                            }

                            $images_ids = $product->get_gallery_image_ids();

                            if (!empty($images_ids)) {
                                if (!empty($images_ids[0])) {
                                    if ($thumbnail_image = wp_get_attachment_image_src($images_ids[0], $thumbnail_size)) {
                                        $thumbnail_src[] = $thumbnail_image[0];
                                    }
                                }
                            }
                            ?>

                            <?php if ($settings['theme'] == 'one'): ?>
                                <div class="product-thumbnail">
                                    <a href="<?= esc_url($product->get_permalink()); ?>" title="<?= $product->get_title(); ?>" class="<?= (count($thumbnail_src) > 1) ? 'has-images' : 'has-image'; ?>">
                                        <?php foreach ($thumbnail_src as $thumbnail_key => $thumbnail): ?>
                                            <img <?= ws_image_lazy_load_attr(); ?> src="<?= $thumbnail; ?>" class="<?= ($thumbnail_key == 1) ? 'secondary-thumbnail' : 'primary-thumbnail'; ?>" alt="<?= $product->get_title(); ?>" />
                                        <?php endforeach; ?>
                                    </a>

                                    <?php if ($product->is_on_sale() && $settings['show_onsale_badge'] == 'yes') : ?>
                                        <?php if (get_theme_mod('show-price-discount-percentage', 1)): ?>
                                            <span class="onsale"><?= ws_get_on_sale_product_discount_percentage($product); ?></span>
                                        <?php else: ?>
                                            <?php echo apply_filters('woocommerce_sale_flash', '<span class="onsale">' . esc_html__('Sale!', 'woocommerce') . '</span>', $post, $product); ?>
                                        <?php endif; ?>
                                    <?php endif; ?>

                                    <div class="icon-toggles">
                                        <?php if (class_exists('YITH_Woocompare_Frontend')): ?>
                                            <?= do_shortcode("[yith_compare_button]"); ?>
                                        <?php endif; ?>

                                        <?php if (function_exists('ws_wccm_add_button')): ?>
                                            <?php ws_wccm_add_button($product->get_id()); ?>
                                        <?php endif; ?>

                                        <?php if (function_exists('tinvwl_shortcode_addtowishlist')): ?>
                                            <div class="add-to-wishlist">
                                                <?= do_shortcode("[ti_wishlists_addtowishlist loop=yes product_id=" . $product->get_id() . "]"); ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>

                                    <?php if (function_exists('woo_variation_swatches')): ?>
                                        <?= ws_wc_display_product_loop_color_attributes($product, 'pa_color'); ?>
                                    <?php endif; ?>
                                </div>
                                <div class="product-details">
                                    <h3 class="product-title">
                                        <a href="<?= esc_url(get_permalink($product->get_id())); ?>" title="<?= $product->get_title(); ?>">
                                            <?php if (!empty($settings['title_limit'])) : ?>
                                                <?= ws_get_content_limit($product->get_title(), 80, true); ?>
                                            <?php else : ?>
                                                <?= $product->get_title(); ?>
                                            <?php endif; ?>
                                        </a>
                                    </h3>

                                    <?php include get_template_directory() . '/templates/product-loop/product-footer.php' ?>
                                </div>
                            <?php endif; ?>
                            <?php if ($settings['theme'] == 'two'): ?>
                                <?php if ($settings['show_cart_button'] == 'yes'): ?>
                                    <div class="floating-add-to-cart">
                                        <div class="floating-add-to-cart-con">
                                            <?php if ($product->is_purchasable() && $product->is_in_stock()) : ?>
                                                <a href="<?= esc_url($product->add_to_cart_url()); ?>" data-quantity="1" data-product_id="<?= $product->get_id(); ?>" class="button add_to_cart_button ajax_add_to_cart" aria-label="<?= __('Add to cart', 'webafra-store'); ?>">
                                                    <i class="cart-icon fa-light fa-cart-plus"></i>
                                                    <i class="added-icon fa-light fa-cart-circle-check"></i>
                                                    <span><?= __('Add to cart', 'webafra-store'); ?></span>
                                                </a>
                                            <?php else : ?>
                                                <a href="<?= esc_url($product->add_to_cart_url()); ?>" data-quantity="1" data-product_id="<?= $product->get_id(); ?>" class="button add_to_cart_button">
                                                    <i class="fa-light fa-cart-circle-exclamation"></i>
                                                    <span><?= __('Read more', 'webafra-store'); ?></span>
                                                </a>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <div class="product-thumbnail">
                                    <a href="<?= esc_url($product->get_permalink()); ?>" title="<?= $product->get_title(); ?>" class="<?= (count($thumbnail_src) > 1) ? 'has-images' : 'has-image'; ?>">
                                        <?php foreach ($thumbnail_src as $thumbnail_key => $thumbnail): ?>
                                            <img <?= ws_image_lazy_load_attr(); ?> src="<?= $thumbnail; ?>" class="<?= ($thumbnail_key == 1) ? 'secondary-thumbnail' : 'primary-thumbnail'; ?>" alt="<?= $product->get_title(); ?>" />
                                        <?php endforeach; ?>
                                    </a>

                                    <?php if ($product->is_on_sale() && $settings['show_onsale_badge'] == 'yes') : ?>
                                        <?php if (get_theme_mod('show-price-discount-percentage', 1)): ?>
                                            <span class="onsale"><?= ws_get_on_sale_product_discount_percentage($product); ?></span>
                                        <?php else: ?>
                                            <?php echo apply_filters('woocommerce_sale_flash', '<span class="onsale">' . esc_html__('Sale!', 'woocommerce') . '</span>', $post, $product); ?>
                                        <?php endif; ?>
                                    <?php endif; ?>

                                    <div class="icon-toggles">
                                        <?php if (class_exists('YITH_Woocompare_Frontend')): ?>
                                            <?= do_shortcode("[yith_compare_button]"); ?>
                                        <?php endif; ?>

                                        <?php if (function_exists('ws_wccm_add_button')): ?>
                                            <?php ws_wccm_add_button($product->get_id()); ?>
                                        <?php endif; ?>

                                        <?php if (function_exists('tinvwl_shortcode_addtowishlist')): ?>
                                            <div class="add-to-wishlist">
                                                <?= do_shortcode("[ti_wishlists_addtowishlist loop=yes product_id=" . $product->get_id() . "]"); ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>

                                    <?php if (function_exists('woo_variation_swatches')): ?>
                                        <?= ws_wc_display_product_loop_color_attributes($product, 'pa_color'); ?>
                                    <?php endif; ?>
                                </div>
                                <div class="product-details">
                                    <h3 class="product-title">
                                        <a href="<?= esc_url(get_permalink($product->get_id())); ?>" title="<?= $product->get_title(); ?>">
                                            <?php if (!empty($settings['title_limit'])) : ?>
                                                <?= ws_get_content_limit($product->get_title(), 80, true); ?>
                                            <?php else : ?>
                                                <?= $product->get_title(); ?>
                                            <?php endif; ?>
                                        </a>
                                    </h3>

                                    <?php if ($has_sale_price_dates_to): ?>
                                        <div class="count-down">
                                            <?php $sale_price_dates_to = get_post_meta($product->get_id(), get_theme_mod('shop-product-offer-end-date-key', '_woo_product_offerslider_timer'), true); ?>
                                            <?php if (!empty($sale_price_dates_to)): ?>
                                                <ul class="countDown-timer" id="display_timer_<?= $this->get_id(); ?>-<?= $product->get_id(); ?>"></ul>

                                                <i class="fa-light fa-alarm-clock"></i>

                                                <script>
                                                    product_timer_init(
                                                        '#display_timer_<?= $this->get_id(); ?>-<?= $product->get_id(); ?>',
                                                        '<?= date('Y-m-d', $sale_price_dates_to); ?>',
                                                        '<?= __('Offer is finished', 'webafra-store'); ?>',
                                                    );
                                                </script>
                                            <?php endif; ?>
                                        </div>
                                    <?php endif; ?>

                                    <div class="product-footer">
                                        <?php if ($settings['show_rating'] == 'yes'): ?>
                                            <div class="product-rating">
                                                <span><?= $product->get_average_rating(); ?></span>
                                                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 16 16">
                                                    <path fill="#EEC200" d="M6.701 1.654a1.64 1.64 0 012.979 0l1.053 2.197c.242.505.71.855 1.25.936l2.354.352c1.362.204 1.906 1.928.92 2.918l-1.703 1.71c-.391.393-.57.96-.478 1.514l.403 2.415c.232 1.398-1.192 2.464-2.41 1.804l-2.106-1.14a1.618 1.618 0 00-1.545 0L5.312 15.5c-1.218.66-2.642-.406-2.41-1.803l.403-2.416a1.743 1.743 0 00-.478-1.514l-1.704-1.71c-.985-.99-.441-2.714.92-2.918l2.355-.352c.541-.081 1.009-.431 1.25-.936l1.053-2.197z"></path>
                                                </svg>
                                            </div>
                                        <?php endif; ?>

                                        <div class="product-price">
                                            <?php if ($price_html = $product->get_price_html()) : ?>
                                                <span class="price"><?php echo $price_html; ?></span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>

                    <?php if ($settings['additional_item_end_enable'] == 'yes'): ?>
                        <div class="additional-item end">
                            <<?= !empty($settings['additional_item_end_link']['url']) ? 'a target="' . ($settings['additional_item_end_link']['is_external'] == true ? '_blank' : '_self') . '" href="' . esc_url($settings['additional_item_end_link']['url']) . '"' : 'div'; ?> class="additional-item-link">
                                <?php if (!empty($settings['additional_item_end_label'])): ?>
                                    <div class="additional-item-label">
                                        <span><?= $settings['additional_item_end_label']; ?></span>
                                    </div>
                                <?php endif; ?>
                                <span class="more-icon"><i class="fa-solid fa-arrow-left-long"></i></span>
                            </<?= !empty($settings['additional_item_end_link']['url']) ? 'a' : 'div'; ?>>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            <script>
                jQuery(document).ready(function($) {
                    products_carousel_init({
                        id: `#webafra_store_elementor_products_carousel-<?= $this->get_id(); ?>`,
                        rtl: <?= (int) is_rtl(); ?>,
                        margin: <?= !empty($settings['carousel_gap']) ? (int) $settings['carousel_gap']['size'] : 0; ?>,
                        stage_padding: <?= !empty($settings['stage_padding']) ? (int) $settings['stage_padding'] : 0; ?>,
                        limit: <?= !empty($settings['carousel_limit']) ? (int) $settings['carousel_limit'] : 4; ?>,
                        limit_tablet: <?= !empty($settings['carousel_limit_tablet']) ? (int) $settings['carousel_limit_tablet'] : 4; ?>,
                        limit_mobile: <?= !empty($settings['carousel_limit_mobile']) ? (int) $settings['carousel_limit_mobile'] : 4; ?>,
                        show_nav: <?= $settings['show_nav'] == 'yes' ? 1 : 0; ?>,
                        nav_container: '#custom-nav-<?= $this->get_id(); ?>',
                        autoplay: <?= $settings['autoplay'] == 'yes' ? 1 : 0; ?>,
                        auto_width: <?= $settings['auto_width'] == 'yes' ? 1 : 0; ?>,
                        slide_progress_id: '#slide-progress-<?= $this->get_id(); ?>',
                    });
                });
            </script>
<?php endif;
    }
}
