<?php

use Elementor\Controls_Manager;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Repeater;
use Elementor\Widget_Base;

if (!defined('ABSPATH')) {
    exit('No direct access allowed.');
}

class Webafra_Store_Elementor_Widget_Products_Tabs extends Widget_Base
{
    public function get_name()
    {
        return 'webafra_store_elementor_products_tabs';
    }

    public function get_title()
    {
        return esc_html__('Products (Tabs)', 'webafra-store');
    }

    public function get_icon()
    {
        return 'eicon-products';
    }

    public function get_script_depends()
    {
        return [];
    }

    public function get_style_depends()
    {
        return [
            'webafra-store-products-tabs',
        ];
    }

    public function get_categories()
    {
        return ['webafra-store-widgets'];
    }

    protected function register_controls()
    {
        $this->start_controls_section('query', [
            'label' => esc_html__('Query', 'webafra-store'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $tab_queries = new Repeater();

        $tab_queries->add_control('tab_title', [
            'label' => esc_html__('Title', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => esc_html__('Default title', 'webafra-store'),
            'placeholder' => esc_html__('Type your title here', 'webafra-store'),
        ]);

        $tab_queries->add_control('source_type', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Source', 'webafra-store'),
            'options' => [
                'latest' => esc_html__('Latest', 'webafra-store'),
                'featured' => esc_html__('Featured', 'webafra-store'),
                'sale' => esc_html__('Sale', 'webafra-store'),
                'sale_dates' => esc_html__('Sale with dates', 'webafra-store'),
            ],
            'default' => ['latest'],
        ]);

        $categories = new \WP_Term_Query([
            'taxonomy' => 'product_cat',
            'hide_empty' => 0,
            'number' => 0,
        ]);

        $categories_new = [];

        if (!empty($categories->get_terms())) {
            foreach (array_values($categories->get_terms()) as $cat) {
                $categories_new[$cat->term_id] = $cat->name;
            }
        }

        $tab_queries->add_control('include_cat', [
            'label' => esc_html__('Include', 'webafra-store'),
            'type' => Controls_Manager::SELECT2,
            'label_block' => true,
            'multiple' => true,
            'options' => $categories_new,
        ]);

        $tab_queries->add_control('exclude_cat', [
            'label' => esc_html__('Exclude', 'webafra-store'),
            'type' => Controls_Manager::SELECT2,
            'label_block' => true,
            'multiple' => true,
            'options' => $categories_new,
        ]);

        $tab_queries->add_control('limit', [
            'type' => Controls_Manager::NUMBER,
            'label' => esc_html__('Limit', 'webafra-store'),
            'default' => 5,
            'min' => 1,
            'max' => 50
        ]);

        $tab_queries->add_control('order_by', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Order by', 'webafra-store'),
            'options' => [
                'name' => esc_html__('Name', 'webafra-store'),
                'rand' => esc_html__('Rand', 'webafra-store'),
            ],
            'default' => 'name',
        ]);

        $tab_queries->add_control('order', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Order', 'webafra-store'),
            'options' => [
                'DESC' => esc_html__('DESC', 'webafra-store'),
                'ASC' => esc_html__('ASC', 'webafra-store'),
            ],
            'default' => 'ASC',
            'separator' => 'after'
        ]);

        $this->add_control('tab_queries', [
            'label' => esc_html__('Tab queries', 'webafra-store'),
            'type' => Controls_Manager::REPEATER,
            'fields' => $tab_queries->get_controls(),
            'default' => [
                [
                    'tab_title' => esc_html__('Default title', 'webafra-store'),
                ]
            ],
            'title_field' => '{{{tab_title}}}',
        ]);

        $this->add_control('thumbnail_size', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Thumbnail size', 'webafra-store'),
            'multiple' => false,
            'options' => ws_get_thumbnail_sizes(),
            'default' => 'woocommerce_thumbnail',
        ]);

        $this->add_control('thumbnail_size_custom_width', [
            'label' => esc_html__('Width', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => esc_html__('Width', 'webafra-store'),
            'condition' => [
                'thumbnail_size' => 'custom'
            ]
        ]);

        $this->add_control('thumbnail_size_custom_height', [
            'label' => esc_html__('Height', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => esc_html__('Height', 'webafra-store'),
            'condition' => [
                'thumbnail_size' => 'custom'
            ]
        ]);

        $this->add_control('show_rating', [
            'label' => esc_html__('Show rating', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('show_onsale_badge', [
            'label' => esc_html__('Show onsale badge', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->end_controls_section();

        $this->start_controls_section('section_style', [
            'label' => esc_html__('Section', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
        ]);

        $this->add_control('section_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper' => 'background-color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('height', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Height', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper' => 'height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('section_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_group_control(Group_Control_Box_Shadow::get_type(), [
            'name' => 'section_shadow',
            'selector' => '{{WRAPPER}} .ws-products-tabs .products-wrapper',
        ]);

        $this->add_responsive_control('section_columns', [
            'label' => esc_html__('Columns', 'webafra-store'),
            'type' => Controls_Manager::NUMBER,
            'default' => 4,
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop' => 'width: calc((100% / {{VALUE}}) - ((({{VALUE}} - 1) / {{VALUE}}) * {{section_columns_gap.SIZE}}{{section_columns_gap.UNIT}}));'
            ]
        ]);

        $this->add_responsive_control('section_columns_gap', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Columns gap', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .tab-content' => 'gap: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->end_controls_section();

        $this->start_controls_section('product_style', [
            'label' => esc_html__('Product', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
        ]);

        $this->add_responsive_control('product_title_alignment', [
            'label' => esc_html__('Title alignment', 'webafra-store'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => esc_html__('Left', 'webafra-store'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => esc_html__('Center', 'webafra-store'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => esc_html__('Right', 'webafra-store'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'right',
            'toggle' => true,
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-title' => 'text-align: {{VALUE}};',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Title', 'webafra-store'),
            'name' => 'product_title_typography',
            'selector' => '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-title a'
        ]);

        $this->add_control('product_title_color', [
            'label' => esc_html__('Title color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-title a' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('product_price_heading', [
            'type' => Controls_Manager::HEADING,
            'label' => esc_html__('Price', 'webafra-store'),
        ]);

        $this->add_responsive_control('product_price_alignment', [
            'label' => esc_html__('Price alignment', 'webafra-store'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'flex-end' => [
                    'title' => esc_html__('Left', 'webafra-store'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => esc_html__('Center', 'webafra-store'),
                    'icon' => 'eicon-text-align-center',
                ],
                'flex-start' => [
                    'title' => esc_html__('Right', 'webafra-store'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'right',
            'toggle' => true,
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-bottom .product-price .price' => 'align-items: {{VALUE}};',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Price', 'webafra-store'),
            'name' => 'product_price_typography',
            'selector' => '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-bottom .product-price .price'
        ]);

        $this->add_control('product_price_color', [
            'label' => esc_html__('Price color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-bottom .product-price .price' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Price del', 'webafra-store'),
            'name' => 'product_price_del_typography',
            'selector' => '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-bottom .product-price .price del'
        ]);

        $this->add_control('product_price_del_color', [
            'label' => esc_html__('Price del color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-bottom .product-price .price del' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('product_price_height', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Price height', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-bottom .product-price' => 'height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('product_rating_heading', [
            'type' => Controls_Manager::HEADING,
            'label' => esc_html__('Rating', 'webafra-store'),
            'condition' => [
                'show_rating' => 'yes',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Rating typography', 'webafra-store'),
            'name' => 'product_rating_number_typography',
            'selector' => '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-bottom span',
            'condition' => [
                'show_rating' => 'yes',
            ],
        ]);

        $this->add_control('product_rating_number_color', [
            'label' => esc_html__('Rating color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'condition' => [
                'show_rating' => 'yes',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-bottom span' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('product_rating_icon_size', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Rating icon size', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'condition' => [
                'show_rating' => 'yes',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-details .product-bottom svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_control('product_onsale_badge_hr', [
            'type' => Controls_Manager::DIVIDER,
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_onsale_badge_heading', [
            'label' => esc_html__('Onsale badge', 'webafra-store'),
            'type' => Controls_Manager::HEADING,
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Typography', 'webafra-store'),
            'name' => 'product_onsale_badge_typography',
            'selector' => '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-thumbnail span.onsale',
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_onsale_badge_color', [
            'label' => esc_html__('Color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-thumbnail span.onsale' => 'color: {{VALUE}};',
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_onsale_badge_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-thumbnail span.onsale' => 'background-color: {{VALUE}};',
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_responsive_control('product_onsale_badge_border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-thumbnail span.onsale' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_responsive_control('product_onsale_badge_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop .product-thumbnail span.onsale' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop' => 'background-color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('product_border', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Border', 'webafra-store'),
            'options' => [
                'none' => esc_html__('None', 'webafra-store'),
                'solid' => esc_html__('Solid', 'webafra-store'),
                'dashed' => esc_html__('Dashed', 'webafra-store')
            ],
            'default' => 'none',
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop' => 'border-style: {{VALUE}};',
            ]
        ]);

        $this->add_control('product_border_color', [
            'label' => esc_html__('Border color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop' => 'border-color: {{VALUE}}',
            ],
            'condition' => ['product_border!' => 'none']
        ]);

        $this->add_responsive_control('product_border_width', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border width', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors' => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop' => 'border-top-width: {{TOP}}{{UNIT}}; border-right-width: {{RIGHT}}{{UNIT}}; border-left-width: {{LEFT}}{{UNIT}}; border-bottom-width: {{BOTTOM}}{{UNIT}};'
            ],
            'separator' => 'after',
            'condition' => ['product_border!' => 'none']
        ]);

        $this->add_responsive_control('product_border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('product_margin', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Margin', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('product_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_group_control(Group_Control_Box_Shadow::get_type(), [
            'name' => 'product_shadow',
            'selector' => '{{WRAPPER}} .ws-products-tabs .products-wrapper .product-loop',
        ]);

        $this->end_controls_section();
    }

    private function get_query(array $tab)
    {
        $query = [
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => $tab['limit'],
            'orderby' => $tab['order_by'],
            'order' => $tab['order'],
        ];

        if ($tab['source_type'] == 'featured') {
            $query['post__in'] = wc_get_featured_product_ids();
        } elseif ($tab['source_type'] == 'sale') {
            $query['post__in'] = wc_get_product_ids_on_sale();
        } elseif ($tab['source_type'] == 'sale_dates') {
            $query['post__in'] = wc_get_product_ids_on_sale();
            $query['meta_query'] = [
                'relation' => 'AND',
                [
                    'key' => '_sale_price_dates_to',
                    'value' => current_time('timestamp'),
                    'compare' => '>=',
                    'type' => 'numeric'
                ],
            ];
            $query['meta_key'] = '_sale_price_dates_to';
            $query['orderby'] = 'meta_value_num';
        }

        if (!empty($tab['include_cat']) && is_array($tab['include_cat'])) {
            $query['tax_query'] = [
                'relation' => 'AND',
                [
                    'taxonomy' => 'product_cat',
                    'field' => 'id',
                    'terms' => array_map('intval', (array) $tab['include_cat']),
                    'operator' => 'IN'
                ]
            ];
        }

        if (!empty($tab['exclude_cat']) && is_array($tab['exclude_cat'])) {
            if (!isset($query['tax_query'])) {
                $query['tax_query'] = [
                    'relation' => 'AND',
                ];
            }

            $query['tax_query'][] = [
                'taxonomy' => 'product_cat',
                'field'  => 'id',
                'terms' => array_map('intval', (array) $tab['exclude_cat']),
                'operator' => 'NOT IN',
            ];
        }

        return $query;
    }

    private function get_thumbnail_size()
    {
        $settings = $this->get_settings_for_display();

        $thumbnail_size = $settings['thumbnail_size'];

        if ($settings['thumbnail_size'] == 'custom' && (!empty($settings['thumbnail_size_custom_width']) && !empty($settings['thumbnail_size_custom_height']))) {
            $thumbnail_size = [
                $settings['thumbnail_size_custom_width'],
                $settings['thumbnail_size_custom_height']
            ];
        }

        return $thumbnail_size;
    }

    private function get_thumbnail_src(int $product_id)
    {
        $thumbnail_size = $this->get_thumbnail_size();
        $thumbnail_src = wc_placeholder_img_src($thumbnail_size);

        if ($thumbnail_id = get_post_thumbnail_id($product_id)) {
            if ($thumbnail_image = wp_get_attachment_image_src($thumbnail_id, $thumbnail_size)) {
                $thumbnail_src = $thumbnail_image[0];
            }
        }

        return $thumbnail_src;
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        if (empty($settings['tab_queries'])) {
            return;
        }

        echo '<div class="ws-products-tabs" id="ws-products-tabs-' . $this->get_id() . '">';

        echo '<ul class="tabs">';
        foreach ($settings['tab_queries'] as $tab_key => $tab) {
            $onclick = "change_products_tab('" . $this->get_id() . "','" . $tab['_id'] . "');";

            echo '<li id="' . $tab['_id'] . '" class="tab-' . $tab['_id'] . ($tab_key == 0 ? ' active' : '') . '" onclick="' . $onclick . '">';
            if (!empty($tab['tab_title'])) {
                echo $tab['tab_title'];
            }
            echo '</li>';
        }
        echo '</ul>';

        echo '<div class="products-wrapper">';

        foreach ($settings['tab_queries'] as $tab_key => $tab) {
            $tab_query = $this->get_query($tab);
            $products = new WP_Query($tab_query);

            echo '<div class="tab-content tab-content-' . $tab['_id'] . ($tab_key == 0 ? ' active' : '') . '">';

            if (!empty($products->get_posts())) {
                foreach ($products->get_posts() as $post) {
                    $product = wc_get_product($post);

                    echo '<div class="product-loop product-type-' . $product->get_type() . '">';

                    echo '<div class="product-thumbnail">';

                    echo '<a href="' . esc_url($product->get_permalink()) . '" title="' . $product->get_title() . '"><img ' . ws_image_lazy_load_attr() . ' src="' . $this->get_thumbnail_src($product->get_id()) . '" alt="' . $product->get_title() . '" /></a>';

                    if ($product->is_on_sale() && $settings['show_onsale_badge'] == 'yes') :
                        if (get_theme_mod('show-price-discount-percentage', 1)):
                            echo '<span class="onsale">' . ws_get_on_sale_product_discount_percentage($product) . '</span>';
                        else:
                            echo apply_filters('woocommerce_sale_flash', '<span class="onsale">' . esc_html__('Sale!', 'woocommerce') . '</span>', $post, $product);
                        endif;
                    endif;

                    echo '</div>';

                    echo '<div class="product-details">';

                    echo '<h3 class="product-title"><a href="' . esc_url(get_permalink($product->get_id())) . '" title="' . $product->get_title() . '">' . (!empty($settings['title_limit']) ? ws_get_content_limit($product->get_title(), 80, true) : $product->get_title()) . '</a></h3>';

                    echo '<div class="product-bottom">';

                    echo '<div class="product-rating">';
                    echo '<span>' . $product->get_average_rating() . '</span>';
                    echo '<svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 16 16"><path fill="#EEC200" d="M6.701 1.654a1.64 1.64 0 012.979 0l1.053 2.197c.242.505.71.855 1.25.936l2.354.352c1.362.204 1.906 1.928.92 2.918l-1.703 1.71c-.391.393-.57.96-.478 1.514l.403 2.415c.232 1.398-1.192 2.464-2.41 1.804l-2.106-1.14a1.618 1.618 0 00-1.545 0L5.312 15.5c-1.218.66-2.642-.406-2.41-1.803l.403-2.416a1.743 1.743 0 00-.478-1.514l-1.704-1.71c-.985-.99-.441-2.714.92-2.918l2.355-.352c.541-.081 1.009-.431 1.25-.936l1.053-2.197z"></path></svg>';
                    echo '</div>';

                    echo '<div class="product-price">';
                    if ($price_html = $product->get_price_html()) {
                        echo '<span class="price">' . $price_html . '</span>';
                    }
                    echo '</div>';

                    echo '</div>';

                    echo '</div>';

                    echo '</div>';
                }
            }

            echo '</div>';
        }

        echo '</div>';

        echo '</div>';
    }
}
