<?php

use Elementor\Controls_Manager;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Widget_Base;

if (!defined('ABSPATH')) {
    exit('No direct access allowed.');
}

class Webafra_Store_Elementor_Widget_Products_Grid extends Widget_Base
{
    public function get_name()
    {
        return 'webafra_store_elementor_products_grid';
    }

    public function get_title()
    {
        return esc_html__('Products (Grid)', 'webafra-store');
    }

    public function get_icon()
    {
        return 'eicon-products';
    }

    public function get_script_depends()
    {
        return [];
    }

    public function get_style_depends()
    {
        return [
            'webafra-store-products-grid',
        ];
    }

    public function get_categories()
    {
        return ['webafra-store-widgets'];
    }

    protected function register_controls()
    {
        $this->start_controls_section('query', [
            'label' => esc_html__('Query', 'webafra-store'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $this->add_control('source_type', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Source', 'webafra-store'),
            'options' => [
                'latest' => esc_html__('Latest', 'webafra-store'),
                'featured' => esc_html__('Featured', 'webafra-store'),
                'sale' => esc_html__('Sale', 'webafra-store'),
                'sale_dates' => esc_html__('Sale with dates', 'webafra-store'),
            ],
            'default' => ['latest'],
        ]);

        $categories = new WP_Term_Query([
            'taxonomy' => 'product_cat',
            'hide_empty' => 0,
            'number' => 0,
        ]);

        $categories_new = [];

        if (count($categories->get_terms())) {
            foreach (array_values($categories->get_terms()) as $cat) {
                $categories_new[$cat->term_id] = $cat->name;
            }
        }

        $this->add_control('include_category', [
            'label' => esc_html__('Include', 'webafra-store'),
            'type' => Controls_Manager::SELECT2,
            'label_block' => true,
            'multiple' => true,
            'options' => $categories_new,
            'default' => [],
        ]);

        $this->add_control('limit', [
            'label' => esc_html__('Limit', 'webafra-store'),
            'type' => Controls_Manager::NUMBER,
            'default' => 8,
        ]);

        $this->add_control('thumbnail_size', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Thumbnail size', 'webafra-store'),
            'multiple' => false,
            'options' => ws_get_thumbnail_sizes(),
            'default' => 'woocommerce_thumbnail',
        ]);

        $this->add_control('thumbnail_size_custom_width', [
            'label' => esc_html__('Width', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => esc_html__('Width', 'webafra-store'),
            'condition' => [
                'thumbnail_size' => 'custom'
            ]
        ]);

        $this->add_control('thumbnail_size_custom_height', [
            'label' => esc_html__('Height', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => esc_html__('Height', 'webafra-store'),
            'condition' => [
                'thumbnail_size' => 'custom'
            ]
        ]);

        $this->add_control('show_rating', [
            'label' => esc_html__('Show rating', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('show_onsale_badge', [
            'label' => esc_html__('Show onsale badge', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->end_controls_section();

        $this->start_controls_section('section_style', [
            'label' => esc_html__('Section', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
        ]);

        $this->add_responsive_control('section_columns', [
            'label' => esc_html__('Columns', 'webafra-store'),
            'type' => Controls_Manager::NUMBER,
            'default' => 4,
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop' => 'width: calc((100% / {{VALUE}}) - ((({{VALUE}} - 1) / {{VALUE}}) * 0px));'
            ]
        ]);

        $this->add_control('section_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid' => 'background-color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('height', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Height', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid' => 'height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-grid' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('section_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-grid' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_group_control(Group_Control_Box_Shadow::get_type(), [
            'name' => 'section_shadow',
            'selector' => '{{WRAPPER}} .ws-products-grid',
        ]);

        $this->end_controls_section();

        $this->start_controls_section('product_style', [
            'label' => esc_html__('Product', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
        ]);

        $this->add_responsive_control('product_title_alignment', [
            'label' => esc_html__('Title alignment', 'webafra-store'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => esc_html__('Left', 'webafra-store'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => esc_html__('Center', 'webafra-store'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => esc_html__('Right', 'webafra-store'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'right',
            'toggle' => true,
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-title' => 'text-align: {{VALUE}};',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Title', 'webafra-store'),
            'name' => 'product_title_typography',
            'selector' => '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-title a'
        ]);

        $this->add_control('product_title_color', [
            'label' => esc_html__('Title color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-title a' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('product_price_heading', [
            'type' => Controls_Manager::HEADING,
            'label' => esc_html__('Price', 'webafra-store'),
        ]);

        $this->add_responsive_control('price_alignment', [
            'label' => esc_html__('Price alignment', 'webafra-store'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'flex-end' => [
                    'title' => esc_html__('Left', 'webafra-store'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => esc_html__('Center', 'webafra-store'),
                    'icon' => 'eicon-text-align-center',
                ],
                'flex-start' => [
                    'title' => esc_html__('Right', 'webafra-store'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'right',
            'toggle' => true,
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-bottom .product-price .price' => 'align-items: {{VALUE}};',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Price', 'webafra-store'),
            'name' => 'product_price_typography',
            'selector' => '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-bottom .product-price .price'
        ]);

        $this->add_control('product_price_color', [
            'label' => esc_html__('Price color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-bottom .product-price .price' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Price del', 'webafra-store'),
            'name' => 'product_price_del_typography',
            'selector' => '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-bottom .product-price .price del'
        ]);

        $this->add_control('product_price_del_color', [
            'label' => esc_html__('Price del color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-bottom .product-price .price del' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('price_height', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Price height', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-bottom .product-price' => 'height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('product_rating_heading', [
            'type' => Controls_Manager::HEADING,
            'label' => esc_html__('Rating', 'webafra-store'),
            'condition' => [
                'show_rating' => 'yes',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Rating typography', 'webafra-store'),
            'name' => 'product_rating_number_typography',
            'selector' => '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-bottom .product-rating span',
            'condition' => [
                'show_rating' => 'yes',
            ],
        ]);

        $this->add_control('product_rating_number_color', [
            'label' => esc_html__('Rating color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'condition' => [
                'show_rating' => 'yes',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-bottom .product-rating span' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('product_rating_icon_size', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Rating icon size', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'condition' => [
                'show_rating' => 'yes',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-details .product-bottom .product-rating svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_control('product_onsale_badge_hr', [
            'type' => Controls_Manager::DIVIDER,
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_onsale_badge_heading', [
            'label' => esc_html__('Onsale badge', 'webafra-store'),
            'type' => Controls_Manager::HEADING,
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Typography', 'webafra-store'),
            'name' => 'product_onsale_badge_typography',
            'selector' => '{{WRAPPER}} .ws-products-grid .product-loop .product-thumbnail span.onsale',
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_onsale_badge_color', [
            'label' => esc_html__('Color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-thumbnail span.onsale' => 'color: {{VALUE}};',
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_onsale_badge_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-thumbnail span.onsale' => 'background-color: {{VALUE}};',
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_responsive_control('product_onsale_badge_border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-thumbnail span.onsale' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_responsive_control('product_onsale_badge_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-grid .product-loop .product-thumbnail span.onsale' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
            'condition' => [
                'show_onsale_badge' => 'yes'
            ]
        ]);

        $this->add_control('product_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop' => 'background-color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('product_border', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Border', 'webafra-store'),
            'options' => [
                'none' => esc_html__('None', 'webafra-store'),
                'solid' => esc_html__('Solid', 'webafra-store'),
                'dashed' => esc_html__('Dashed', 'webafra-store')
            ],
            'default' => 'none',
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop' => 'border-style: {{VALUE}};',
            ]
        ]);

        $this->add_control('product_border_color', [
            'label' => esc_html__('Border color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop' => 'border-color: {{VALUE}}',
            ],
            'condition' => ['product_border!' => 'none']
        ]);

        $this->add_responsive_control('product_border_width', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border width', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors' => [
                '{{WRAPPER}} .ws-products-grid .product-loop' => 'border-top-width: {{TOP}}{{UNIT}}; border-right-width: {{RIGHT}}{{UNIT}}; border-left-width: {{LEFT}}{{UNIT}}; border-bottom-width: {{BOTTOM}}{{UNIT}};'
            ],
            'separator' => 'after',
            'condition' => ['product_border!' => 'none']
        ]);

        $this->add_responsive_control('product_border_radius', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Border radius', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-grid .product-loop' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('product_margin', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Margin', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-grid .product-loop' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_responsive_control('product_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-products-grid .product-loop' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->add_group_control(Group_Control_Box_Shadow::get_type(), [
            'name' => 'product_shadow',
            'selector' => '{{WRAPPER}} .ws-products-grid .product-loop',
        ]);

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        $args = [
            'post_type' => 'product',
            'post_status' => 'publish',
            'ignore_sticky_posts' => 1,
            'posts_per_page' => !empty($settings['limit']) ? $settings['limit'] : 8,
            'orderby' => 'date',
            'order' => 'desc',
        ];

        if ($settings['source_type'] == 'featured') {
            $args['post__in'] = wc_get_featured_product_ids();
        } elseif ($settings['source_type'] == 'sale') {
            $args['post__in'] = wc_get_product_ids_on_sale();
        } elseif ($settings['source_type'] == 'sale_dates') {
            $args['post__in'] = wc_get_product_ids_on_sale();
            $args['meta_query'] = [
                'relation' => 'AND',
                [
                    'key' => '_sale_price_dates_to',
                    'value' => current_time('timestamp'),
                    'compare' => '>=',
                    'type' => 'numeric'
                ],
            ];
            $args['meta_key'] = '_sale_price_dates_from';
            $args['orderby'] = 'meta_value_num';
        }

        if (!empty($settings['include_category'])) {
            $args['tax_query'] = [
                [
                    'taxonomy' => 'product_cat',
                    'field' => 'term_id',
                    'terms' => $settings['include_category'],
                    'operator' => 'IN' // 'IN', 'NOT IN', 'AND'
                ]
            ];
        }

        $products = new WP_Query($args); ?>

        <?php if (count($products->get_posts())) : ?>
            <div class="ws-products-grid">
                <?php foreach ($products->get_posts() as $post) : $product = wc_get_product($post); ?>
                    <div class="product-loop product-type-<?= $product->get_type(); ?>">
                        <?php
                        $thumbnail_size = $settings['thumbnail_size'];

                        if ($settings['thumbnail_size'] == 'custom' && (!empty($settings['thumbnail_size_custom_width']) && !empty($settings['thumbnail_size_custom_height']))) {
                            $thumbnail_size = [
                                $settings['thumbnail_size_custom_width'],
                                $settings['thumbnail_size_custom_height']
                            ];
                        }

                        $thumbnail_src = wc_placeholder_img_src($thumbnail_size);

                        if ($thumbnail_id = get_post_thumbnail_id($product->get_id())) {
                            if ($thumbnail_image = wp_get_attachment_image_src($thumbnail_id, $thumbnail_size)) {
                                $thumbnail_src = $thumbnail_image[0];
                            }
                        }
                        ?>
                        <div class="product-thumbnail">
                            <a href="<?= esc_url($product->get_permalink()); ?>" title="<?= $product->get_title(); ?>">
                                <img <?= ws_image_lazy_load_attr(); ?> src="<?= $thumbnail_src; ?>" alt="<?= $product->get_title(); ?>" />
                            </a>
                            <?php if ($product->is_on_sale() && $settings['show_onsale_badge'] == 'yes') : ?>
                                <?php if (get_theme_mod('show-price-discount-percentage', 1)): ?>
                                    <span class="onsale"><?= ws_get_on_sale_product_discount_percentage($product); ?></span>
                                <?php else: ?>
                                    <?php echo apply_filters('woocommerce_sale_flash', '<span class="onsale">' . esc_html__('Sale!', 'woocommerce') . '</span>', $post, $product); ?>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                        <div class="product-details">
                            <h3 class="product-title">
                                <a href="<?= esc_url(get_permalink($product->get_id())); ?>" title="<?= $product->get_title(); ?>">
                                    <?php if (!empty($settings['title_limit'])) : ?>
                                        <?= ws_get_content_limit($product->get_title(), 80, true); ?>
                                    <?php else : ?>
                                        <?= $product->get_title(); ?>
                                    <?php endif; ?>
                                </a>
                            </h3>

                            <div class="product-bottom">
                                <div class="product-rating">
                                    <span><?= $product->get_average_rating(); ?></span>
                                    <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 16 16">
                                        <path fill="#EEC200" d="M6.701 1.654a1.64 1.64 0 012.979 0l1.053 2.197c.242.505.71.855 1.25.936l2.354.352c1.362.204 1.906 1.928.92 2.918l-1.703 1.71c-.391.393-.57.96-.478 1.514l.403 2.415c.232 1.398-1.192 2.464-2.41 1.804l-2.106-1.14a1.618 1.618 0 00-1.545 0L5.312 15.5c-1.218.66-2.642-.406-2.41-1.803l.403-2.416a1.743 1.743 0 00-.478-1.514l-1.704-1.71c-.985-.99-.441-2.714.92-2.918l2.355-.352c.541-.081 1.009-.431 1.25-.936l1.053-2.197z"></path>
                                    </svg>
                                </div>
                                <div class="product-price">
                                    <?php if ($price_html = $product->get_price_html()) : ?>
                                        <span class="price"><?php echo $price_html; ?></span>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
<?php endif;
    }
}
