<?php

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Widget_Base;

if (!defined('ABSPATH')) {
    exit('No direct access allowed.');
}

class Webafra_Store_Elementor_Widget_Posts extends Widget_Base
{
    public function get_name()
    {
        return 'webafra_store_elementor_posts';
    }

    public function get_title()
    {
        return esc_html__('Posts', 'webafra-store');
    }

    public function get_icon()
    {
        return 'eicon-post-list';
    }

    public function get_script_depends()
    {
        return [
            'webafra-store-owl.carousel.min',
        ];
    }

    public function get_style_depends()
    {
        return [
            'webafra-store-owl.carousel.min',
            'webafra-store-owl.theme.default.min',
            'webafra-store-posts',
        ];
    }

    public function get_categories()
    {
        return ['webafra-store-widgets'];
    }

    protected function register_controls()
    {
        $this->start_controls_section('settings', [
            'label' => esc_html__('Settings', 'webafra-store'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $this->add_control('title', [
            'label' => esc_html__('Title', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => esc_html__('Title', 'webafra-store'),
        ]);

        $this->add_control('enable_section', [
            'label' => esc_html__('Enable section', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('thumbnail_size', [
            'type' => Controls_Manager::SELECT,
            'label' => esc_html__('Thumbnail size', 'webafra-store'),
            'multiple' => false,
            'options' => ws_get_thumbnail_sizes(),
            'default' => 'thumbnail',
        ]);

        $this->add_control('thumbnail_size_custom_width', [
            'label' => esc_html__('Width', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => esc_html__('Width', 'webafra-store'),
            'condition' => [
                'thumbnail_size' => 'custom'
            ]
        ]);

        $this->add_control('thumbnail_size_custom_height', [
            'label' => esc_html__('Height', 'webafra-store'),
            'type' => Controls_Manager::TEXT,
            'default' => esc_html__('Height', 'webafra-store'),
            'condition' => [
                'thumbnail_size' => 'custom'
            ]
        ]);

        $this->add_control('show_date_badge', [
            'label' => esc_html__('Show date badge', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->end_controls_section();

        $this->start_controls_section('query', [
            'label' => esc_html__('Query', 'webafra-store'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $categories = new WP_Term_Query([
            'taxonomy' => 'category',
            'hide_empty' => 0,
            'number' => 0,
        ]);

        $categories_new = [];

        if (count($categories->get_terms())) {
            foreach (array_values($categories->get_terms()) as $cat) {
                $categories_new[$cat->term_id] = $cat->name;
            }
        }

        $this->add_control('include_category', [
            'label' => esc_html__('Include', 'webafra-store'),
            'type' => Controls_Manager::SELECT2,
            'label_block' => true,
            'multiple' => true,
            'options' => $categories_new,
            'default' => [],
        ]);

        $this->add_control('limit', [
            'label' => esc_html__('Limit', 'webafra-store'),
            'type' => Controls_Manager::NUMBER,
            'default' => 8,
        ]);

        $this->end_controls_section();

        $this->start_controls_section('carousel', [
            'label' => esc_html__('Carousel', 'webafra-store'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $this->add_responsive_control('carousel_limit', [
            'label' => esc_html__('Carousel limit', 'webafra-store'),
            'type' => Controls_Manager::NUMBER,
            'default' => 4,
        ]);

        $this->add_control('autoplay', [
            'label' => esc_html__('Autoplay', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->add_control('show_nav', [
            'label' => esc_html__('Show nav', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
        ]);

        $this->end_controls_section();

        $this->start_controls_section('section_style', [
            'label' => esc_html__('Section', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
            'condition' => [
                'enable_section' => 'yes'
            ]
        ]);

        $this->add_responsive_control('section_title', [
            'label' => esc_html__('Title alignment', 'webafra-store'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => esc_html__('Left', 'webafra-store'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => esc_html__('Center', 'webafra-store'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => esc_html__('Right', 'webafra-store'),
                    'icon' => 'eicon-text-align-right',
                ],
            ],
            'default' => 'right',
            'toggle' => true,
            'selectors' => [
                '{{WRAPPER}} .ws-posts .posts-head .head-title' => 'text-align: {{VALUE}};',
            ],
        ]);

        $this->add_control('section_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-posts.section-box' => 'background-color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('height', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Height', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-posts.section-box' => 'height: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('section_padding', [
            'type' => Controls_Manager::DIMENSIONS,
            'label' => esc_html__('Padding', 'webafra-store'),
            'size_units' => ['px', 'em', '%'],
            'selectors'  => [
                '{{WRAPPER}} .ws-posts.section-box' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
            ],
            'default' => [
                'top' => '',
                'right' => '',
                'bottom' => '',
                'left' => '',
                'unit' => 'px',
                'isLinked' => true,
            ],
        ]);

        $this->end_controls_section();

        $this->start_controls_section('date_badge_style', [
            'label' => esc_html__('Date badge', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
            'condition' => [
                'show_date_badge' => 'yes'
            ]
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Typography', 'webafra-store'),
            'name' => 'date_badge_typography',
            'selector' => '{{WRAPPER}} .ws-posts .posts-wrapper .post-loop .post-thumbnail .post-date'
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Month typography', 'webafra-store'),
            'name' => 'date_badge_month_typography',
            'selector' => '{{WRAPPER}} .ws-posts .posts-wrapper .post-loop .post-thumbnail .post-date .post-date-month'
        ]);

        $this->add_control('date_badge_background_color', [
            'label' => esc_html__('Background color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-posts .posts-wrapper .post-loop .post-thumbnail .post-date' => 'background-color: {{VALUE}}',
            ],
        ]);

        $this->add_control('date_badge_color', [
            'label' => esc_html__('Color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'default' => '',
            'selectors' => [
                '{{WRAPPER}} .ws-posts .posts-wrapper .post-loop .post-thumbnail .post-date' => 'color: {{VALUE}}',
            ],
        ]);

        $this->end_controls_section();

        $this->start_controls_section('title_style', [
            'label' => esc_html__('Title', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Typography', 'webafra-store'),
            'name' => 'title_typography',
            'selector' => '{{WRAPPER}} .ws-posts .posts-wrapper .post-loop .post-content .post-title a'
        ]);

        $this->start_controls_tabs('title_color_tab');

        $this->start_controls_tab('title_normal_tab', [
            'label' => esc_html__('Normal', 'webafra-store'),
        ]);

        $this->add_control('title_color', [
            'label' => esc_html__('Color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .ws-posts .posts-wrapper .post-loop .post-content .post-title a' => 'color: {{VALUE}}',
            ],
        ]);

        $this->end_controls_tab();

        $this->start_controls_tab('title_hover_tab', [
            'label' => esc_html__('Hover', 'webafra-store'),
        ]);

        $this->add_control('title_hover_color', [
            'label' => esc_html__('Color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .ws-posts .posts-wrapper .post-loop .post-content .post-title a:hover' => 'color: {{VALUE}}',
            ],
        ]);

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section('description_style', [
            'label' => esc_html__('Description', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Typography', 'webafra-store'),
            'name' => 'description_typography',
            'selector' => '{{WRAPPER}} .ws-posts .posts-wrapper .post-loop .post-content .post-desc'
        ]);

        $this->add_control('description_color', [
            'label' => esc_html__('Color', 'webafra-store'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .ws-posts .posts-wrapper .post-loop .post-content .post-desc' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_responsive_control('description_alignment', [
            'label' => esc_html__('Alignment', 'webafra-store'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'left' => [
                    'title' => esc_html__('Left', 'webafra-store'),
                    'icon' => 'eicon-text-align-left',
                ],
                'center' => [
                    'title' => esc_html__('Center', 'webafra-store'),
                    'icon' => 'eicon-text-align-center',
                ],
                'right' => [
                    'title' => esc_html__('Right', 'webafra-store'),
                    'icon' => 'eicon-text-align-right',
                ],
                'justify' => [
                    'title' => esc_html__('Justify', 'webafra-store'),
                    'icon' => 'eicon-text-align-justify',
                ],
            ],
            'default' => 'justify',
            'toggle' => true,
            'selectors' => [
                '{{WRAPPER}} .ws-posts .posts-wrapper .post-loop .post-content .post-desc' => 'text-align: {{VALUE}};',
            ],
        ]);

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        $args = [
            'post_type' => 'post',
            'post_status' => 'publish',
            'ignore_sticky_posts' => 1,
            'posts_per_page' => !empty($settings['limit']) ? $settings['limit'] : 8,
            'orderby' => 'date',
            'order' => 'desc',
        ];

        if (!empty($settings['include_category'])) {
            $args['tax_query'] = [
                [
                    'taxonomy' => 'category',
                    'field' => 'term_id',
                    'terms' => $settings['include_category'],
                    'operator' => 'IN' // 'IN', 'NOT IN', 'AND'
                ]
            ];
        }

        $posts = new WP_Query($args); ?>

        <?php if (count($posts->get_posts())) : ?>
            <div class="ws-posts<?= $settings['enable_section'] == 'yes' ? ' section-box' : ''; ?>">
                <div class="posts-head">
                    <div class="head-title">
                        <?php if (!empty($settings['title'])): ?>
                            <h2><?= $settings['title']; ?></h2>
                        <?php endif; ?>
                    </div>
                    <?php if ($settings['show_nav'] == 'yes'): ?>
                        <div id="custom-nav-<?= $this->get_id(); ?>" class="custom-nav"></div>
                    <?php endif; ?>
                </div>
                <div id="webafra_store_elementor_posts-<?= $this->get_id(); ?>" class="posts-wrapper owl-carousel owl-theme">
                    <?php foreach ($posts->get_posts() as $post) : ?>
                        <div class="post-loop">
                            <div class="floating-read-more">
                                <a href="<?= esc_url(get_permalink($post)); ?>" title="<?= get_the_title($post); ?>">
                                    <i class="fa-light fa-file-lines"></i>
                                    <span><?= __('Read more', 'webafra-store'); ?></span>
                                </a>
                            </div>
                            <div class="post-thumbnail">
                                <div class="thumbnail-img">
                                    <?php
                                    $thumbnail_size = $settings['thumbnail_size'];

                                    if ($settings['thumbnail_size'] == 'custom' && (!empty($settings['thumbnail_size_custom_width']) && !empty($settings['thumbnail_size_custom_height']))) {
                                        $thumbnail_size = [
                                            $settings['thumbnail_size_custom_width'],
                                            $settings['thumbnail_size_custom_height']
                                        ];
                                    }

                                    $thumbnail_src = ws_get_thumbnail_placeholder();
                                    if ($thumbnail_id = get_post_thumbnail_id($post->ID)) {
                                        if ($thumbnail_image = wp_get_attachment_image_src($thumbnail_id, $thumbnail_size)) {
                                            $thumbnail_src = $thumbnail_image[0];
                                        }
                                    }
                                    ?>
                                    <a href="<?= esc_url(get_permalink($post)); ?>" title="<?= get_the_title($post); ?>">
                                        <img <?= ws_image_lazy_load_attr(); ?> src="<?= $thumbnail_src; ?>" alt="<?= get_the_title($post); ?>" />
                                    </a>
                                </div>

                                <?php if ($settings['show_date_badge'] == 'yes'): ?>
                                    <div class="post-date">
                                        <div class="post-date-day"><?= get_the_date('d', $post); ?></div>
                                        <div class="post-date-month"><?= get_the_date('F', $post); ?></div>
                                    </div>
                                <?php endif; ?>

                                <?php if ($primary_cat = ws_get_post_primary_cat($post->ID)): ?>
                                    <div class="post-cat">
                                        <a title="<?= $primary_cat->name; ?>" href="<?= get_term_link($primary_cat); ?>"><?= $primary_cat->name; ?></a>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="post-content">
                                <h3 class="post-title">
                                    <a title="<?= get_the_title($post); ?>" href="<?= esc_url(get_permalink()); ?>" rel="bookmark"><?= get_the_title($post); ?></a>
                                </h3>
                                <?php if (!empty($post->post_content)): ?>
                                    <div class="post-desc"><?= ws_get_content_limit($post->post_content, 100, true); ?></div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <script>
                (function($) {
                    jQuery(`#webafra_store_elementor_posts-<?= $this->get_id(); ?>`).owlCarousel({
                        items: <?= !empty($settings['carousel_limit']) ? $settings['carousel_limit'] : 4; ?>,
                        margin: 10,
                        rtl: <?= (int) is_rtl(); ?>,
                        lazyLoad: 1,
                        responsive: {
                            0: {
                                items: <?= !empty($settings['carousel_limit_mobile']) ? $settings['carousel_limit_mobile'] : 4; ?>,
                            },
                            768: {
                                items: <?= !empty($settings['carousel_limit_tablet']) ? $settings['carousel_limit_tablet'] : 4; ?>,
                            },
                            1024: {
                                items: <?= !empty($settings['carousel_limit']) ? $settings['carousel_limit'] : 4; ?>,
                            },
                            1440: {
                                items: <?= !empty($settings['carousel_limit']) ? $settings['carousel_limit'] : 4; ?>
                            }
                        },
                        nav: <?= $settings['show_nav'] == 'yes' ? 1 : 0; ?>,
                        navContainer: '#custom-nav-<?= $this->get_id(); ?>',
                        navText: [
                            '<svg aria-label="Previous" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#c8c8c8" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M9 18l6-6-6-6"/></svg>',
                            '<svg aria-label="Next" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#c8c8c8" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M15 18l-6-6 6-6"/></svg>',
                        ],
                        dots: 0,
                        autoplay: <?= $settings['autoplay'] == 'yes' ? 1 : 0; ?>,
                        autoplayTimeout: 10000,
                    });
                })(jQuery);
            </script>
<?php endif;
    }
}
