<?php

use Elementor\Controls_Manager;
use Elementor\Widget_Base;

if (!defined('ABSPATH')) {
    exit('No direct access allowed.');
}

class Webafra_Store_Elementor_Widget_MegaMenu extends Widget_Base
{
    public function get_name()
    {
        return 'webafra_store_elementor_megamenu';
    }

    public function get_title()
    {
        return esc_html__('Megamenu', 'webafra-store');
    }

    public function get_icon()
    {
        return 'eicon-products';
    }

    public function get_script_depends()
    {
        return [];
    }

    public function get_style_depends()
    {
        return [
            'webafra-store-megamenu',
        ];
    }

    public function get_categories()
    {
        return ['webafra-store-widgets'];
    }

    protected function register_controls()
    {
        $this->start_controls_section('query', [
            'label' => esc_html__('Query', 'webafra-store'),
            'tab' => Controls_Manager::TAB_CONTENT,
        ]);

        $categories = new WP_Term_Query([
            'taxonomy' => 'product_cat',
            'hide_empty' => 0,
            'number' => 0,
        ]);

        $categories_new = [];

        if (count($categories->get_terms())) {
            foreach (array_values($categories->get_terms()) as $cat) {
                $categories_new[$cat->term_id] = $cat->name;
            }
        }

        $this->add_control('include_category', [
            'label' => esc_html__('Include', 'webafra-store'),
            'type' => Controls_Manager::SELECT2,
            'label_block' => true,
            'multiple' => true,
            'options' => $categories_new,
            'default' => [],
        ]);

        $this->add_control('exclude_category', [
            'label' => esc_html__('Exclude', 'webafra-store'),
            'type' => Controls_Manager::SELECT2,
            'label_block' => true,
            'multiple' => true,
            'options' => $categories_new,
            'default' => [],
        ]);

        $this->add_control('show_count', [
            'label' => esc_html__('Show count', 'webafra-store'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'no',
        ]);

        $this->end_controls_section();

        $this->start_controls_section('style', [
            'label' => esc_html__('Style', 'webafra-store'),
            'tab' => Controls_Manager::TAB_STYLE,
        ]);

        $this->add_responsive_control('dropdown_width', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Dropdown width', 'webafra-store'),
            'size_units' => ['px', '%'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-mega-menu .mega-menu-dropdown' => 'width: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->add_responsive_control('absolute_right_position', [
            'type' => Controls_Manager::SLIDER,
            'label' => esc_html__('Right position', 'webafra-store'),
            'size_units' => ['px', '%', 'vw'],
            'range' => [
                'px' => [
                    'min' => 0,
                    'max' => 2000,
                ],
                'vw' => [
                    'min' => -200,
                    'max' => 200,
                ],
                '%' => [
                    'min' => 0,
                    'max' => 100,
                ]
            ],
            'default' => [
                'unit' => 'px',
                'size' => '',
            ],
            'selectors' => [
                '{{WRAPPER}} .ws-mega-menu .mega-menu-dropdown' => 'right: {{SIZE}}{{UNIT}};'
            ]
        ]);

        $this->end_controls_section();
    }

    private function get_child(int $parent)
    {
        $query = [
            'taxonomy' => 'product_cat',
            'hide_empty' => 0,
            'number' => 0,
            'parent' => $parent,
        ];

        $parents = new WP_Term_Query($query);

        return $parents->get_terms();
    }

    private function get_thumbnail_src(int $term_id, string $thumbnail_size = 'thumbnail')
    {
        $thumbnail_src = ws_get_thumbnail_placeholder();
        if ($thumbnail_id = get_term_meta($term_id, 'thumbnail_id', true)) {
            if ($thumbnail_image = wp_get_attachment_image_src($thumbnail_id, $thumbnail_size)) {
                $thumbnail_src = $thumbnail_image[0];
            }
        }
        return $thumbnail_src;
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        $query = [
            'taxonomy' => 'product_cat',
            'hide_empty' => 0,
            'number' => 0,
            'parent' => 0,
        ];

        if (!empty($settings['include_category']) && is_array($settings['include_category'])) {
            $query['include'] = $settings['include_category'];
        }

        if (!empty($settings['exclude_category']) && is_array($settings['exclude_category'])) {
            $query['exclude_category'] = $settings['exclude_category'];
        }

        $parents = new WP_Term_Query($query);
?>
        <nav class="ws-mega-menu" id="ws-mega-menu-<?= $this->get_id(); ?>">
            <button class="mega-menu-button">
                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 20 20">
                    <path stroke="#2E353C" stroke-linecap="round" stroke-linejoin="round" stroke-miterlimit="10" stroke-width="1.5" d="M14.167 8.333h1.666c1.667 0 2.5-.833 2.5-2.5V4.167c0-1.667-.833-2.5-2.5-2.5h-1.666c-1.667 0-2.5.833-2.5 2.5v1.666c0 1.667.833 2.5 2.5 2.5zM4.167 18.333h1.666c1.667 0 2.5-.833 2.5-2.5v-1.666c0-1.667-.833-2.5-2.5-2.5H4.167c-1.667 0-2.5.833-2.5 2.5v1.666c0 1.667.833 2.5 2.5 2.5zM5 8.333a3.333 3.333 0 100-6.666 3.333 3.333 0 000 6.666zM15 18.333a3.333 3.333 0 100-6.666 3.333 3.333 0 000 6.666z"></path>
                </svg>
                <span>دسته‌بندی</span>
                <i class="fa-solid fa-angle-down"></i>
            </button>

            <div class="mega-menu-dropdown">
                <ul class="main-category-tabs">
                    <?php if (!empty($parents->get_terms())): ?>
                        <?php foreach ($parents->get_terms() as $term_key => $term): ?>
                            <li <?= $term_key == 0 ? 'class="active"' : ''; ?> id="term-id-<?= $term->term_id; ?>"><?= $term->name; ?></li>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </ul>
                <div class="main-category-contents">
                    <?php if (!empty($parents->get_terms())): ?>
                        <?php foreach ($parents->get_terms() as $term_key => $term): ?>
                            <div class="main-category-content<?= $term_key == 0 ? ' active' : ''; ?>" id="term-id-<?= $term->term_id; ?>">
                                <?php if (!empty($this->get_child($term->term_id))): ?>
                                    <?php foreach ($this->get_child($term->term_id) as $term_child): ?>
                                        <a href="<?= esc_url(get_term_link($term_child)); ?>" title="<?= $term_child->name; ?>">
                                            <div class="category-thumbnail"><img src="<?= $this->get_thumbnail_src($term_child->term_id); ?>" alt="<?= $term_child->name; ?>" /></div>
                                            <div class="category-details">
                                                <h3 class="category-title"><?= $term_child->name; ?></h3>
                                                <?php if ($settings['show_count'] == 'yes'): ?>
                                                    <div class="count"><?= $term_child->count; ?> <?= __('Product', 'webafra-store'); ?></div>
                                                <?php endif; ?>
                                            </div>
                                        </a>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    <script>
                        jQuery(document).ready(function($) {
                            megamenu_init('ws-mega-menu-<?= $this->get_id(); ?>');
                            megamenu_responsive('ws-mega-menu-<?= $this->get_id(); ?>');
                        });
                    </script>
                </div>

            </div>
        </nav>
<?php
    }
}
