<?php
if (!defined('ABSPATH')) {
   die;
}

if (!class_exists('WFS_Admin_Page')) {
   class WFS_Admin_Page
   {
      public static $ins = null;

      public $request_uri = '';
      public $error = '';
      public $form_status = [];

      public $per_pages_items = ['5', '10', '20', '50', '100', '150', '200'];

      public static function ins()
      {
         if (is_null(self::$ins)) {
            self::$ins = new self();
         }

         return self::$ins;
      }

      public function __construct()
      {
         $this->request_uri = $_SERVER['REQUEST_URI'];

         add_filter('upload_mimes', [$this, 'upload_mime_types']);
         add_filter('admin_title', [$this, 'admin_title'], 10, 2);

         add_action('admin_enqueue_scripts', [$this, 'admin_enqueue_scripts']);
         add_action('admin_menu', [$this, 'register_menu']);

         add_action('wp_dashboard_setup', function () {
            if (current_user_can('manage_options')) {
               wp_add_dashboard_widget('wfs_dashboard_widget', sprintf('<span class="dashboard-widget-heading">%s: %s</span>', __('Fara Search', 'webafra-fara-search'), __('Top keywords', 'webafra-fara-search')), [$this, 'dashboard_widget']);
            }
         });
      }

      public function dashboard_widget()
      {
         $search_keywords = WFS_DB_Queries::get_search_keywords([
            'orderby'      => 'count',
            'per_pages'    => 5
         ]);

         include_once wfs_PP . '/templates/admin/dashboard-widget.php';
      }

      public function admin_enqueue_scripts($hook)
      {
         $version = time();

         wp_enqueue_media();

         wp_enqueue_script('jquery');

         wp_enqueue_style('wfs-jquery-ui.min', wfs_PF . 'assets/css/jquery-ui.min.css', [], $version);
         wp_enqueue_script('wfs-jquery-ui.min', wfs_PF . 'assets/js/jquery-ui.min.js', [], $version, false);

         wp_enqueue_script('webafra-fara-search', wfs_PF . 'assets/js/admin.js', ['jquery'], $version, false);

         wp_localize_script('webafra-fara-search', 'wfs_vars', [
            'ajax_url'     => admin_url('admin-ajax.php'),
            'ajax_nonce'   => wp_create_nonce('wfs-ajax-nonce'),
            'is_rtl'       => (int) is_rtl(),
         ]);

         wp_enqueue_style('webafra-fara-search', wfs_PF . 'assets/css/admin.css', [], $version);
      }

      public function upload_mime_types($mimes)
      {
         $mimes['svg']           = 'image/svg+xml';
         $mimes['png']           = 'image/png';
         $mimes['jpg|jpeg|jpe']  = 'image/jpeg';
         $mimes['webp']          = 'image/webp';

         return $mimes;
      }

      /**
       * Change admin pages admin title.
       * 
       */
      public function admin_title($admin_title, $title)
      {
         if (isset($_GET['page']) && $_GET['page'] == 'wfs-search-logs') {
            if (isset($_GET['tab']) && $_GET['tab'] == 'jobs') {
               return sprintf(
                  '%s > %s > %s',
                  get_bloginfo('name'),
                  __('Fara Search', 'webafra-fara-search'),
                  __('Scheduled Events', 'webafra-fara-search')
               );
            } elseif (isset($_GET['tab']) && $_GET['tab'] == 'logs') {
               return sprintf(
                  '%s > %s > %s',
                  get_bloginfo('name'),
                  __('Fara Search', 'webafra-fara-search'),
                  __('Search logs', 'webafra-fara-search')
               );
            } elseif (!isset($_GET['tab']) || (isset($_GET['tab']) && $_GET['tab'] == 'keywords')) {
               return sprintf(
                  '%s > %s > %s',
                  get_bloginfo('name'),
                  __('Fara Search', 'webafra-fara-search'),
                  __('Searched words', 'webafra-fara-search')
               );
            }
         }

         if (isset($_GET['page']) && $_GET['page'] == 'wfs-search-forms') {
            if (isset($_GET['form_id']) && is_numeric($_GET['form_id'])) {
               if (!isset($_GET['tab']) || (isset($_GET['tab']) && $_GET['tab'] == 'general')) {
                  return sprintf(
                     '%s > %s > %s',
                     get_bloginfo('name'),
                     __('Edit ajax search form', 'webafra-fara-search'),
                     __('General', 'webafra-fara-search')
                  );
               } elseif (isset($_GET['tab']) && $_GET['tab'] == 'style') {
                  return sprintf(
                     '%s > %s > %s',
                     get_bloginfo('name'),
                     __('Edit ajax search form', 'webafra-fara-search'),
                     __('Style', 'webafra-fara-search')
                  );
               } elseif (isset($_GET['tab']) && $_GET['tab'] == 'translate') {
                  return sprintf(
                     '%s > %s > %s',
                     get_bloginfo('name'),
                     __('Edit ajax search form', 'webafra-fara-search'),
                     __('Translate', 'webafra-fara-search')
                  );
               }
            }
         }

         return $admin_title;
      }

      public function register_menu()
      {
         add_menu_page(
            __('Fara Search', 'webafra-fara-search'),
            __('Fara Search', 'webafra-fara-search'),
            'manage_options',
            'wfs-general',
            [$this, 'register_menu_callback'],
            'dashicons-search',
            6
         );

         add_submenu_page(
            'wfs-general',
            __('Search forms', 'webafra-fara-search'),
            __('Search forms', 'webafra-fara-search'),
            'manage_options',
            'wfs-search-forms',
            [$this, 'register_forms_callback'],
         );

         add_submenu_page(
            'wfs-general',
            __('Search logs', 'webafra-fara-search'),
            __('Search logs', 'webafra-fara-search'),
            'manage_options',
            'wfs-search-logs',
            [$this, 'register_menu_callback'],
         );

         add_submenu_page(
            'wfs-general',
            __('Settings', 'webafra-fara-search'),
            __('Settings', 'webafra-fara-search'),
            'manage_options',
            'wfs-settings',
            [$this, 'register_settings_callback'],
         );

         unset($GLOBALS['submenu']['wfs-general'][0]);
      }

      private function get_top_search_texts_daily_for_chart(int $form_id = null)
      {
         $top_search_texts = WFS_DB_Queries::get_top_search_texts_daily($form_id);

         $collect_data = [];
         $collect = [];

         if (!empty($top_search_texts)) {
            foreach ($top_search_texts as $text) {
               if (!isset($collect_data[$text->date])) {
                  $collect_data[$text->date] = [
                     'x' => wac_date($text->date, get_option('wff_settings_date_calendar', 'gregorian_calendar'), 'Y/m/d'),
                     'y' => 0,
                  ];
               }
            }

            foreach ($top_search_texts as $text) {
               if (!isset($collect[$text->text])) {
                  $collect[$text->text] = [
                     'label'  => $text->text,
                     'data'   => array_values($collect_data)
                  ];
               }
            }

            foreach ($top_search_texts as $text) {
               $collect[$text->text]['data'] = array_map(function ($data) use ($text) {
                  if ($data['x'] == wac_date($text->date, get_option('wff_settings_date_calendar', 'gregorian_calendar'), 'Y/m/d')) {
                     $data['y'] = $text->count;
                  }
                  return $data;
               }, $collect[$text->text]['data']);
            }
         }

         return array_values($collect);
      }

      private function submit_filter_by_date(string $form_action)
      {
         $start_gregorian = null;

         if (isset($_POST['filter_by_date_start']) && !empty($_POST['filter_by_date_start'])) {
            $start_date = explode('-', $_POST['filter_by_date_start']);

            if (is_array($start_date) && count($start_date) == 3) {
               if (get_option('wfs_settings_calendar', 'gregorian') == 'shamsi_calendar') {
                  $start_gregorian = jalali_to_gregorian($start_date[0], $start_date[1], $start_date[2], '-');
               } elseif (get_option('wfs_settings_calendar', 'gregorian') == 'hijri_calendar') {
                  if (class_exists('uCal2')) {
                     $hijri = new uCal2();
                     $hijri->setLang('ar');
                     $start_gregorian = $hijri->u2g($start_date[2], $start_date[1], $start_date[0]);
                     $start_gregorian = sprintf('%d-%d-%d', $start_gregorian['year'], $start_gregorian['month'], $start_gregorian['day']);
                  }
               } else {
                  $start_gregorian = sprintf('%d-%d-%d', $start_date[0], $start_date[1], $start_date[2]);
               }
            }
         }

         $end_gregorian = null;

         if (isset($_POST['filter_by_date_end']) && !empty($_POST['filter_by_date_end'])) {
            $end_date = explode('-', $_POST['filter_by_date_end']);

            if (is_array($end_date) && count($end_date) == 3) {
               if (get_option('wfs_settings_calendar', 'gregorian') == 'shamsi_calendar') {
                  $end_gregorian = jalali_to_gregorian($end_date[0], $end_date[1], $end_date[2], '-');
               } elseif (get_option('wfs_settings_calendar', 'gregorian') == 'hijri_calendar') {
                  if (class_exists('uCal2')) {
                     $hijri = new uCal2();
                     $hijri->setLang('ar');
                     $end_gregorian = $hijri->u2g($end_date[2], $end_date[1], $end_date[0]);
                     $end_gregorian = sprintf('%d-%d-%d', $end_gregorian['year'], $end_gregorian['month'], $end_gregorian['day']);
                  }
               } else {
                  $end_gregorian = sprintf('%d-%d-%d', $end_date[0], $end_date[1], $end_date[2]);
               }
            }
         }

         if (!is_null($start_gregorian) && !is_null($end_gregorian)) {
            $form_action = "{$form_action}&filter_by_created_at={$start_gregorian},{$end_gregorian}";
         } elseif (!is_null($start_gregorian) && is_null($end_gregorian)) {
            $form_action = "{$form_action}&filter_by_created_at={$start_gregorian},";
         } elseif (is_null($start_gregorian) && !is_null($end_gregorian)) {
            $form_action = "{$form_action}&filter_by_created_at=,{$end_gregorian}";
         } elseif (is_null($start_gregorian) && is_null($end_gregorian)) {
            $form_action = "{$form_action}&filter_by_created_at";
         }

         return $form_action;
      }

      public function register_menu_callback()
      {
         $current_tab = isset($_GET['tab']) && in_array($_GET['tab'], ['logs', 'keywords']) ? $_GET['tab'] : 'keywords';

         $this->request_uri = $_SERVER['REQUEST_URI'];

         $args = [];

         if (isset($_GET['order']) && in_array($_GET['order'], ['ASC', 'DESC'])) {
            $args['order'] = strtoupper($_GET['order']);
         }

         $args['paged'] = 1;

         if (isset($_GET['paged']) && is_numeric($_GET['paged'])) {
            $args['paged'] = $_GET['paged'];
         }

         $args['per_pages'] = get_option('wfs_search_logs_per_pages', 5);

         if (isset($_POST['per_pages_top_action'])) {
            if (isset($_POST['per_pages_top'])) {
               update_option('wfs_search_logs_per_pages', $_POST['per_pages_top']);
               $this->request_uri = "{$this->request_uri}&per_pages={$_POST['per_pages_top']}";
            }

            wp_redirect($this->request_uri);
            exit;
         }

         if (isset($_POST['per_pages_bottom_action'])) {
            if (isset($_POST['per_pages_bottom'])) {
               update_option('wfs_search_logs_per_pages', $_POST['per_pages_bottom']);
               $this->request_uri = "{$this->request_uri}&per_pages={$_POST['per_pages_bottom']}";
            }

            wp_redirect($this->request_uri);
            exit;
         }

         if (isset($_GET['per_pages']) && is_numeric($_GET['per_pages'])) {
            $args['per_pages'] = $_GET['per_pages'];
         }

         if (in_array($current_tab, ['logs', 'keywords'])) {
            $forms_raw = WFS_DB_Queries::get_forms_raw();
         }

         if ($current_tab == 'logs') {
            if (isset($_POST['bulk_items']) && $_POST['bulk_action'] == 'delete_selected') {
               if (is_array($_POST['bulk_items']) && count($_POST['bulk_items'])) {
                  $ids_to_delete = implode(',', array_map('absint', $_POST['bulk_items']));
                  WFS_DB_Queries::delete_search_logs($ids_to_delete);
                  wp_redirect($this->request_uri);
                  exit;
               }
            }

            if (isset($_POST['wfs_search_log_delete_submit']) && (isset($_POST['wfs_search_log_delete_id']) && is_numeric($_POST['wfs_search_log_delete_id']))) {
               WFS_DB_Queries::delete_search_log((int) $_POST['wfs_search_log_delete_id']);
               wp_redirect($this->request_uri);
               exit;
            }

            if ($_SERVER['REQUEST_METHOD'] == 'POST') {
               // Filtering.
               if (isset($_POST['filter_by'])) {
                  $this->request_uri = "{$this->request_uri}&filter_by={$_POST['filter_by']}";
               }

               // Filtering by form id.
               if (isset($_POST['filter_by_form'])) {
                  $this->request_uri = "{$this->request_uri}&filter_by_form={$_POST['filter_by_form']}";
               }

               // Search keyword.
               if (isset($_POST['search'])) {
                  $this->request_uri = "{$this->request_uri}&search={$_POST['search']}";
               }

               // Filtering by date.
               $this->request_uri = $this->submit_filter_by_date($this->request_uri);

               wp_redirect($this->request_uri);
               exit;
            }

            if (isset($_GET['orderby']) && in_array($_GET['orderby'], [
               'text',
               'user_id',
               'has_result',
               'created_at'
            ])) {
               $args['orderby'] = $_GET['orderby'];
            }

            $args['wheres'] = [];

            if (isset($_GET['filter_by']) && in_array($_GET['filter_by'], ['has_result', 'has_no_result'])) {
               $args['wheres']['filter_by'] = $_GET['filter_by'];
            }

            if (isset($_GET['filter_by_created_at'])) {
               $args['wheres']['created_at'] = $_GET['filter_by_created_at'];
            }

            if (isset($_GET['filter_by_form'])) {
               $args['wheres']['form_id'] = (int) $_GET['filter_by_form'];
            }

            if (isset($_GET['search'])) {
               $args['wheres']['search'] = sanitize_text_field($_GET['search']);
            }

            $search_logs = WFS_DB_Queries::get_search_logs($args);
         } elseif ($current_tab == 'keywords') {
            if (isset($_POST['bulk_items']) && $_POST['bulk_action'] == 'delete_selected') {
               if (is_array($_POST['bulk_items']) && count($_POST['bulk_items'])) {
                  $ids_to_delete = implode(',', array_map('absint', $_POST['bulk_items']));
                  WFS_DB_Queries::delete_search_keywords($ids_to_delete);
                  wp_redirect($this->request_uri);
                  exit;
               }
            }

            if (isset($_POST['bulk_items']) && $_POST['bulk_action'] == 'make_selected_special') {
               if (is_array($_POST['bulk_items']) && count($_POST['bulk_items'])) {
                  $ids_to_make = implode(',', array_map('absint', $_POST['bulk_items']));
                  WFS_DB_Queries::make_search_keywords_special($ids_to_make);
                  wp_redirect($this->request_uri);
                  exit;
               }
            }

            if (isset($_POST['bulk_items']) && $_POST['bulk_action'] == 'make_selected_out_of_special') {
               if (is_array($_POST['bulk_items']) && count($_POST['bulk_items'])) {
                  $ids_to_make = implode(',', array_map('absint', $_POST['bulk_items']));
                  WFS_DB_Queries::make_search_keywords_out_of_special($ids_to_make);
                  wp_redirect($this->request_uri);
                  exit;
               }
            }

            if (isset($_POST['bulk_items']) && $_POST['bulk_action'] == 'make_selected_ignore') {
               if (is_array($_POST['bulk_items']) && count($_POST['bulk_items'])) {
                  $ids_to_make = implode(',', array_map('absint', $_POST['bulk_items']));
                  WFS_DB_Queries::make_search_keywords_ignore($ids_to_make);
                  wp_redirect($this->request_uri);
                  exit;
               }
            }

            if (isset($_POST['bulk_items']) && $_POST['bulk_action'] == 'make_selected_out_of_ignore') {
               if (is_array($_POST['bulk_items']) && count($_POST['bulk_items'])) {
                  $ids_to_make = implode(',', array_map('absint', $_POST['bulk_items']));
                  WFS_DB_Queries::make_search_keywords_out_of_ignore($ids_to_make);
                  wp_redirect($this->request_uri);
                  exit;
               }
            }

            // Delete search keyword action.
            if (isset($_POST['wfs_search_keyword_delete_submit']) && (isset($_POST['wfs_search_keyword_delete_id']) && is_numeric($_POST['wfs_search_keyword_delete_id']))) {
               WFS_DB_Queries::delete_search_keyword((int) $_POST['wfs_search_keyword_delete_id']);
               wp_redirect($this->request_uri);
               exit;
            }

            // Edit search keyword action.
            if (isset($_POST['wfs_search_keyword_edit_submit']) && (isset($_POST['wfs_search_keyword_edit_id']) && is_numeric($_POST['wfs_search_keyword_edit_id']))) {
               if (empty($_POST['text']) && empty($_POST['form_id'])) {
                  $this->error = __('Text and form field are required!', 'webafra-fara-search');
               } elseif (empty($_POST['text'])) {
                  $this->error = __('Text field is required!', 'webafra-fara-search');
               } elseif (empty($_POST['form_id'])) {
                  $this->error = __('Form field is required!', 'webafra-fara-search');
               }

               if (isset($_POST['text']) && !empty($_POST['text']) && isset($_POST['form_id']) && is_numeric($_POST['form_id'])) {
                  WFS_DB_Queries::edit_search_keyword((int) $_POST['wfs_search_keyword_edit_id'], [
                     'text'         => $_POST['text'],
                     'form_id'      => $_POST['form_id'],
                     'is_special'   => isset($_POST['is_special']) ? 1 : 0,
                     'is_ignored'   => isset($_POST['is_ignored']) ? 1 : 0,
                     'count'        => isset($_POST['count']) && is_numeric($_POST['count']) ? $_POST['count'] : 0
                  ]);

                  wp_redirect($this->request_uri);
                  exit;
               }
            }

            // Add new keyword action.
            if (isset($_POST['wfs_search_keyword_new_submit'])) {
               if (empty($_POST['text']) && empty($_POST['form_id'])) {
                  $this->error = __('Text and form field are required!', 'webafra-fara-search');
               } elseif (empty($_POST['text'])) {
                  $this->error = __('Text field is required!', 'webafra-fara-search');
               } elseif (empty($_POST['form_id'])) {
                  $this->error = __('Form field is required!', 'webafra-fara-search');
               }

               if (isset($_POST['text']) && !empty($_POST['text']) && isset($_POST['form_id']) && is_numeric($_POST['form_id'])) {
                  WFS_DB_Queries::add_search_keyword([
                     'text'         => $_POST['text'],
                     'form_id'      => $_POST['form_id'],
                     'is_special'   => isset($_POST['is_special']) ? 1 : 0,
                     'is_ignored'   => isset($_POST['is_ignored']) ? 1 : 0,
                  ]);
                  wp_redirect($this->request_uri);
                  exit;
               }
            }

            if ($_SERVER['REQUEST_METHOD'] == 'POST') {
               // Filtering.
               if (isset($_POST['filter_by'])) {
                  $this->request_uri = "{$this->request_uri}&filter_by={$_POST['filter_by']}";
               }

               // Filtering by form id.
               if (isset($_POST['filter_by_form'])) {
                  $this->request_uri = "{$this->request_uri}&filter_by_form={$_POST['filter_by_form']}";
               }

               // Search keyword.
               if (isset($_POST['search'])) {
                  $this->request_uri = "{$this->request_uri}&search={$_POST['search']}";
               }

               // Filtering by date.
               $this->request_uri = $this->submit_filter_by_date($this->request_uri);

               wp_redirect($this->request_uri);
               exit;
            }

            if (isset($_GET['orderby']) && in_array($_GET['orderby'], [
               'text',
               'is_special',
               'is_ignored',
               'count',
               'has_result',
               'created_at'
            ])) {
               $args['orderby'] = $_GET['orderby'];
            }

            $args['wheres'] = [];

            if (isset($_GET['filter_by']) && in_array($_GET['filter_by'], ['is_special', 'is_not_special', 'is_ignored', 'is_not_ignored', 'has_result', 'has_no_result'])) {
               $args['wheres']['filter_by'] = $_GET['filter_by'];
            }

            if (isset($_GET['filter_by_created_at'])) {
               $args['wheres']['created_at'] = $_GET['filter_by_created_at'];
            }

            if (isset($_GET['filter_by_form'])) {
               $args['wheres']['form_id'] = (int) $_GET['filter_by_form'];
            }

            if (isset($_GET['search'])) {
               $args['wheres']['search'] = sanitize_text_field($_GET['search']);
            }

            if (isset($_POST['filter_by_form'])) {
               $this->request_uri = "{$this->request_uri}&filter_by_form={$_POST['filter_by_form']}";
            }

            $search_keywords = WFS_DB_Queries::get_search_keywords($args);
         }

         $template = wfs_PP . '/templates/admin/logs.php';

         if (file_exists($template)) {
            require_once $template;
         }
      }

      public function register_forms_callback()
      {
         $form_id = '';
         $form_data = null;

         $this->form_status = [
            'public'    => __('Public', 'webafra-fara-search'),
            'private'   => __('Private', 'webafra-fara-search'),
            'disable'   => __('Disable', 'webafra-fara-search'),
         ];

         if (isset($_GET['form_id']) && is_numeric($_GET['form_id'])) {
            $form_data = WFS_DB_Queries::get_form((int) $_GET['form_id']);

            if ($form_data !== false) {
               $form_id = (int) $form_data->id;
            } else {
               wp_redirect('admin.php?page=wfs-search-forms');
               exit;
            }
         }

         if (empty($form_id)) {
            if (isset($_POST['wfs_create_new_form_submit'])) {
               if (empty($_POST['form_title']) && empty($_POST['form_status'])) {
                  $this->error = __('Form title and status field are required!', 'webafra-fara-search');
               } elseif (empty($_POST['form_title'])) {
                  $this->error = __('Form title field is required!', 'webafra-fara-search');
               } elseif (empty($_POST['form_status'])) {
                  $this->error = __('Form status is required!', 'webafra-fara-search');
               }

               if ((isset($_POST['form_title']) && !empty($_POST['form_title'])) && (isset($_POST['form_status']) && !empty($_POST['form_status']))) {
                  $new_form = WFS_DB_Queries::insert_form($_POST['form_title'], $_POST['form_status']);
                  if ($new_form) {
                     wp_redirect("admin.php?page=wfs-search-forms&form_id={$new_form}&tab=general");
                     exit;
                  }
               }
            }

            if (isset($_POST['bulk_items']) && $_POST['bulk_action'] == 'delete_selected') {
               if (is_array($_POST['bulk_items']) && count($_POST['bulk_items'])) {
                  $ids_to_delete = implode(',', array_map('absint', $_POST['bulk_items']));
                  WFS_DB_Queries::delete_forms($ids_to_delete);
                  wp_redirect($this->request_uri);
                  exit;
               }
            }

            if (isset($_POST['wfs_delete_form_submit']) && (isset($_POST['wfs_delete_form_delete_id']) && is_numeric($_POST['wfs_delete_form_delete_id']))) {
               WFS_DB_Queries::delete_form((int) $_POST['wfs_delete_form_delete_id']);
               wp_redirect($this->request_uri);
               exit;
            }

            if (isset($_POST['per_pages_top_action'])) {
               if (isset($_POST['per_pages_top'])) {
                  update_option('wfs_forms_per_pages', $_POST['per_pages_top']);
                  $this->request_uri = "{$this->request_uri}&per_pages={$_POST['per_pages_top']}";
               }

               wp_redirect($this->request_uri);
               exit;
            }

            if (isset($_POST['per_pages_bottom_action'])) {
               if (isset($_POST['per_pages_bottom'])) {
                  update_option('wfs_forms_per_pages', $_POST['per_pages_bottom']);
                  $this->request_uri = "{$this->request_uri}&per_pages={$_POST['per_pages_bottom']}";
               }

               wp_redirect($this->request_uri);
               exit;
            }

            $args = [];

            $args['paged'] = 1;

            if (isset($_GET['paged']) && is_numeric($_GET['paged'])) {
               $args['paged'] = $_GET['paged'];
            }

            $args['per_pages'] = get_option('wfs_forms_per_pages', 5);

            if (isset($_GET['per_pages']) && is_numeric($_GET['per_pages'])) {
               $args['per_pages'] = $_GET['per_pages'];
            }

            if (isset($_GET['orderby']) && in_array($_GET['orderby'], ['title', 'status', 'created_at'])) {
               $args['orderby'] = $_GET['orderby'];
            }

            if (isset($_GET['order']) && in_array($_GET['order'], ['ASC', 'DESC'])) {
               $args['order'] = strtoupper($_GET['order']);
            }

            $search_forms = WFS_DB_Queries::get_search_forms($args);

            $template = wfs_PP . '/templates/admin/forms-list.php';
         } else {
            $this->request_uri = $_SERVER['REQUEST_URI'];
            $current_tab = isset($_GET['tab']) && in_array($_GET['tab'], ['general', 'style', 'translate']) ? $_GET['tab'] : 'general';

            if ($current_tab == 'general') {
               if (isset($_POST['form_edit_submit'])) {
                  WFS_DB_Queries::update_form((int) $form_id, [
                     'title'  => sanitize_text_field($_POST['form_title']),
                     'status' => $_POST['form_status'],
                  ]);

                  $form_general_options = [
                     'input_placeholder'                    => sanitize_text_field($_POST['input_placeholder']),
                     'input_keyup_delay'                    => sanitize_text_field($_POST['input_keyup_delay']),
                     'form_icon'                            => sanitize_text_field($_POST['form_icon']),
                     'search_autocomplete'                  => isset($_POST['search_autocomplete']) ? '1' : '0',
                     'search_autocomplete_by'               => sanitize_text_field($_POST['search_autocomplete_by']),
                     'default_post_type'                    => sanitize_text_field($_POST['default_post_type']),
                     'show_post_type_product_results_price' => isset($_POST['show_post_type_product_results_price']) ? '1' : '0',
                     'show_post_type_select'                => isset($_POST['show_post_type_select']) ? '1' : '0',
                     'taxonomy_dropdown'                    => sanitize_text_field($_POST['taxonomy_dropdown']),
                     'post_type_select_location'            => sanitize_text_field($_POST['post_type_select_location']),
                     'dropdown_source'                      => sanitize_text_field($_POST['dropdown_source']),

                     // Search result
                     'search_result_post_types'             => isset($_POST['search_result_post_types']) ? implode(',', $_POST['search_result_post_types']) : '',
                     'search_result_limit'                  => intval($_POST['search_result_limit']),
                     'search_result_title_limit'            => intval($_POST['search_result_title_limit']),
                     'search_result_carousel'               => isset($_POST['search_result_carousel']) ? 1 : 0,
                     'search_result_carousel_limit'         => sanitize_text_field($_POST['search_result_carousel_limit']),
                     'search_result_carousel_limit_tablet'  => sanitize_text_field($_POST['search_result_carousel_limit_tablet']),
                     'search_result_carousel_limit_mobile'  => sanitize_text_field($_POST['search_result_carousel_limit_mobile']),
                     'search_result_icon'                   => sanitize_text_field($_POST['search_result_icon']),

                     'search_in_post_type_categories'             => isset($_POST['search_in_post_type_categories']) ? '1' : '0',
                     'search_in_post_type_categories_title'       => sanitize_text_field($_POST['search_in_post_type_categories_title']),
                     'search_in_post_type_categories_limit'       => sanitize_text_field($_POST['search_in_post_type_categories_limit']),
                     'search_in_post_type_categories_hide_empty'  => isset($_POST['search_in_post_type_categories_hide_empty']) ? 1 : 0,
                     'search_in_post_type_categories_exclude_tax' => isset($_POST['search_in_post_type_categories_exclude_tax']) ? implode(',', $_POST['search_in_post_type_categories_exclude_tax']) : '',
                     'search_in_post_type_categories_icon'        => sanitize_text_field($_POST['search_in_post_type_categories_icon']),

                     'settings_related_limit'   => sanitize_text_field($_POST['settings_related_limit']),
                     'settings_related_icon'    => sanitize_text_field($_POST['settings_related_icon']),

                     // Search banner
                     'search_banner_enable'     => isset($_POST['search_banner_enable']) ? 1 : 0,
                     'search_banner_image'      => intval($_POST['search_banner_image']),
                     'search_banner_image_size' => sanitize_text_field($_POST['search_banner_image_size']),
                     'search_banner_url'        => sanitize_url($_POST['search_banner_url']),

                     // User last searches
                     'user_last_searches_enable'   => isset($_POST['user_last_searches_enable']) ? 1 : 0,
                     'user_last_searches_title'    => sanitize_text_field($_POST['user_last_searches_title']),
                     'user_last_searches_icon'     => sanitize_text_field($_POST['user_last_searches_icon']),

                     // Papular searches
                     'papular_searches_enable'        => isset($_POST['papular_searches_enable']) ? 1 : 0,
                     'papular_searches_title'         => sanitize_text_field($_POST['papular_searches_title']),
                     'papular_searches_range_start'   => str_replace('-', '', $_POST['papular_searches_range_start']),
                     'papular_searches_icon'          => sanitize_text_field($_POST['papular_searches_icon']),

                     // Suggested categories
                     'suggested_categories_enable' => isset($_POST['suggested_categories_enable']) ? 1 : 0,
                     'suggested_categories_title'  => sanitize_text_field($_POST['suggested_categories_title']),
                     'suggested_categories_ids'    => isset($_POST['suggested_categories_ids']) ? implode(',', $_POST['suggested_categories_ids']) : '',
                     'suggested_categories_icon'   => sanitize_text_field($_POST['suggested_categories_icon']),
                  ];

                  $this->error = (array) [];

                  // Validate default post type value.
                  if (empty($form_general_options['search_result_post_types'])) {
                     $form_general_options['show_post_type_product_results_price'] = 0;
                     unset($form_general_options['default_post_type']);
                     $this->error['search_result_post_types'] = __('Default post type must be in post types.', 'webafra-fara-search');
                  } else {
                     if (!in_array($form_general_options['default_post_type'], $_POST['search_result_post_types'])) {
                        unset($form_general_options['search_result_post_types']);
                        unset($form_general_options['default_post_type']);
                        $this->error['search_result_post_types'] = __('Default post type must be in post types.', 'webafra-fara-search');
                     }

                     if (!in_array('product', $_POST['search_result_post_types'])) {
                        $form_general_options['show_post_type_product_results_price'] = 0;
                     }
                  }

                  $form_general_options['use_other_search_results_page'] = isset($_POST['use_other_search_results_page']) ? 1 : 0;

                  if ($form_general_options['use_other_search_results_page']) {
                     if (!empty($_POST['form_submit_redirect_to'])) {
                        $check_post = get_page_by_path((string) $_POST['form_submit_redirect_to'], OBJECT);
                        if (is_null($check_post)) {
                           unset($form_general_options['form_submit_redirect_to']);
                           $this->error['form_submit_redirect_to'] = __('Search result page is not exists!', 'webafra-fara-search');
                        } else {
                           $form_general_options['form_submit_redirect_to']      = (int) $check_post->ID;
                           $form_general_options['form_submit_redirect_to_slug'] = (string) $_POST['form_submit_redirect_to'];
                           $form_general_options['search_query_var']             = sanitize_key($_POST['search_query_var']);
                           $form_general_options['category_query_var']           = !empty($_POST['category_query_var']) ? sanitize_key($_POST['category_query_var']) : 'category_id';

                           if (empty($form_general_options['search_query_var'])) {
                              unset($form_general_options['search_query_var']);
                              $this->error['search_query_var'] = __('Search query var is required!', 'webafra-fara-search');
                           }

                           if ($form_general_options['search_query_var'] == 's') {
                              unset($form_general_options['search_query_var']);
                              $this->error['search_query_var'] = __('Search query var can not use `s` for custom redirect!', 'webafra-fara-search');
                           }

                           $form_general_options['post_type_query_var'] = sanitize_key($_POST['post_type_query_var']);

                           if (empty($form_general_options['post_type_query_var'])) {
                              unset($form_general_options['post_type_query_var']);
                              $this->error['post_type_query_var'] = __('Post type query var is required!', 'webafra-fara-search');
                           }

                           if ($form_general_options['post_type_query_var'] == 'post_type') {
                              unset($form_general_options['post_type_query_var']);
                              $this->error['post_type_query_var'] = __('Post type query var can not use `post_type` for custom redirect!', 'webafra-fara-search');
                           }
                        }
                     } else {
                        $form_general_options['use_other_search_results_page'] = 0;
                        $form_general_options['form_submit_redirect_to']      = null;
                        $form_general_options['form_submit_redirect_to_slug'] = null;
                        $form_general_options['search_query_var']             = 's';
                        $form_general_options['post_type_query_var']          = 'post_type';
                     }
                  } else {
                     $form_general_options['form_submit_redirect_to']      = null;
                     $form_general_options['form_submit_redirect_to_slug'] = null;
                     $form_general_options['search_query_var']             = 's';
                     $form_general_options['post_type_query_var']          = 'post_type';
                  }

                  // Save layout.
                  $form_general_options['layout'] = sanitize_text_field($_POST['layout']);
                  $form_general_options['layout_popup_type'] = sanitize_text_field($_POST['layout_popup_type']);
                  $form_general_options['layout_popup_action_button'] = sanitize_text_field($_POST['layout_popup_action_button']);
                  $form_general_options['layout_popup_action_button_icon'] = sanitize_text_field($_POST['layout_popup_action_button_icon']);
                  $form_general_options['layout_popup_action_button_text'] = sanitize_text_field($_POST['layout_popup_action_button_text']);
                  $form_general_options['layout_popup_button_tooltip_text'] = sanitize_text_field($_POST['layout_popup_button_tooltip_text']);
                  $form_general_options['layout_popup_action_class'] = sanitize_text_field($_POST['layout_popup_action_class']);

                  $file       = wfs_PP . "/backups/form-settings-{$form_id}.json";
                  $saved_json = file_get_contents($file);
                  $saved_arr  = json_decode($saved_json, true);

                  foreach ($form_general_options as $opt_key => $opt) {
                     $saved_arr[$opt_key] = $opt;
                  }

                  file_put_contents($file, wp_json_encode($saved_arr));

                  WFS_DB_Queries::update_form_option((int) $form_id, $form_general_options);

                  if (empty($this->error)) {
                     wp_redirect($this->request_uri . '&success=1');
                     exit;
                  }
               }

               $banner_image = false;

               if (is_numeric($form_data->form_options['search_banner_image'])) {
                  $banner_image = wp_get_attachment_image_src($form_data->form_options['search_banner_image'], 'thumbnail');
               }

               $form_icon = false;

               if (is_numeric($form_data->form_options['form_icon'])) {
                  $form_icon = wp_get_attachment_image_src($form_data->form_options['form_icon'], ['32', '32']);
               }

               $search_result_icon = false;

               if (is_numeric($form_data->form_options['search_result_icon'])) {
                  $search_result_icon = wp_get_attachment_image_src($form_data->form_options['search_result_icon'], ['32', '32']);
               }

               $search_in_post_type_categories_icon = false;

               if (is_numeric($form_data->form_options['search_in_post_type_categories_icon'])) {
                  $search_in_post_type_categories_icon = wp_get_attachment_image_src($form_data->form_options['search_in_post_type_categories_icon'], ['32', '32']);
               }

               $settings_related_icon = false;

               if (is_numeric($form_data->form_options['settings_related_icon'])) {
                  $settings_related_icon = wp_get_attachment_image_src($form_data->form_options['settings_related_icon'], ['32', '32']);
               }

               $user_last_searches_icon = false;

               if (is_numeric($form_data->form_options['user_last_searches_icon'])) {
                  $user_last_searches_icon = wp_get_attachment_image_src($form_data->form_options['user_last_searches_icon'], ['32', '32']);
               }

               $papular_searches_icon = false;

               if (is_numeric($form_data->form_options['papular_searches_icon'])) {
                  $papular_searches_icon = wp_get_attachment_image_src($form_data->form_options['papular_searches_icon'], ['32', '32']);
               }

               $suggested_categories_icon = false;

               if (is_numeric($form_data->form_options['suggested_categories_icon'])) {
                  $suggested_categories_icon = wp_get_attachment_image_src($form_data->form_options['suggested_categories_icon'], ['32', '32']);
               }

               $taxonomies = [];

               if (!empty($form_data->form_options['search_result_post_types'])) {
                  $post_types_arr = explode(',', $form_data->form_options['search_result_post_types']);

                  foreach ($post_types_arr as $post_type) {
                     $post_type_taxonomies_check = get_taxonomies([
                        'object_type' => [$post_type]
                     ], 'objects');

                     if (!empty($post_type_taxonomies_check)) {
                        $taxonomies = array_merge($taxonomies, $post_type_taxonomies_check);
                     }
                  }
               }

               $suggested_categories_ids_to_opt = [];
               if (!empty($form_data->form_options['suggested_categories_ids'])) {
                  $categories = new \WP_Term_Query([
                     'taxonomy'     => array_keys($taxonomies),
                     'hide_empty'   => 0,
                     'number'       => 0,
                     'include'      => explode(',', $form_data->form_options['suggested_categories_ids'])
                  ]);

                  if (count($categories->get_terms())) {
                     foreach (array_values($categories->get_terms()) as $cat) {
                        $tax_name = get_taxonomy($cat->taxonomy)->labels->name;
                        $suggested_categories_ids_to_opt[$cat->term_id] = $cat->name . ' (' . $tax_name . ')';
                     }
                  }
               }
            } elseif ($current_tab == 'style') {
               if (isset($_POST['form_edit_submit'])) {

                  $arguments = [
                     'style_overlay_enable'                    => isset($_POST['style_overlay_enable']) ? 1 : 0,
                     'style_overlay_color'                     => sanitize_text_field($_POST['style_overlay_color']),

                     'layout_popup_type_floating_position'           => sanitize_text_field($_POST['layout_popup_type_floating_position']),
                     'layout_popup_type_floating_right_left'         => sanitize_text_field($_POST['layout_popup_type_floating_right_left']),
                     'layout_popup_type_floating_right_left_tablet'  => sanitize_text_field($_POST['layout_popup_type_floating_right_left_tablet']),
                     'layout_popup_type_floating_right_left_mobile'  => sanitize_text_field($_POST['layout_popup_type_floating_right_left_mobile']),
                     'layout_popup_type_floating_top_bottom'         => sanitize_text_field($_POST['layout_popup_type_floating_top_bottom']),
                     'layout_popup_type_floating_top_bottom_tablet'  => sanitize_text_field($_POST['layout_popup_type_floating_top_bottom_tablet']),
                     'layout_popup_type_floating_top_bottom_mobile'  => sanitize_text_field($_POST['layout_popup_type_floating_top_bottom_mobile']),

                     'layout_popup_action_button_position'           => sanitize_text_field($_POST['layout_popup_action_button_position']),
                     'layout_popup_action_button_right_left'         => sanitize_text_field($_POST['layout_popup_action_button_right_left']),
                     'layout_popup_action_button_right_left_tablet'  => sanitize_text_field($_POST['layout_popup_action_button_right_left_tablet']),
                     'layout_popup_action_button_right_left_mobile'  => sanitize_text_field($_POST['layout_popup_action_button_right_left_mobile']),
                     'layout_popup_action_button_top_bottom'         => sanitize_text_field($_POST['layout_popup_action_button_top_bottom']),
                     'layout_popup_action_button_top_bottom_tablet'  => sanitize_text_field($_POST['layout_popup_action_button_top_bottom_tablet']),
                     'layout_popup_action_button_top_bottom_mobile'  => sanitize_text_field($_POST['layout_popup_action_button_top_bottom_mobile']),

                     'layout_popup_action_button_text_font_size'        => sanitize_text_field($_POST['layout_popup_action_button_text_font_size']),
                     'layout_popup_action_button_text_font_size_tablet' => sanitize_text_field($_POST['layout_popup_action_button_text_font_size_tablet']),
                     'layout_popup_action_button_text_font_size_mobile' => sanitize_text_field($_POST['layout_popup_action_button_text_font_size_mobile']),

                     'layout_popup_action_button_text_color'         => sanitize_text_field($_POST['layout_popup_action_button_text_color']),
                     'layout_popup_action_button_background_color'   => sanitize_text_field($_POST['layout_popup_action_button_background_color']),

                     'layout_popup_action_button_border_radius'         => sanitize_text_field($_POST['layout_popup_action_button_border_radius']),
                     'layout_popup_action_button_border_radius_tablet'  => sanitize_text_field($_POST['layout_popup_action_button_border_radius_tablet']),
                     'layout_popup_action_button_border_radius_mobile'  => sanitize_text_field($_POST['layout_popup_action_button_border_radius_mobile']),

                     'layout_popup_action_button_padding_left'          => sanitize_text_field($_POST['layout_popup_action_button_padding_left']),
                     'layout_popup_action_button_padding_bottom'        => sanitize_text_field($_POST['layout_popup_action_button_padding_bottom']),
                     'layout_popup_action_button_padding_right'         => sanitize_text_field($_POST['layout_popup_action_button_padding_right']),
                     'layout_popup_action_button_padding_top'           => sanitize_text_field($_POST['layout_popup_action_button_padding_top']),
                     'layout_popup_action_button_padding_left_tablet'   => sanitize_text_field($_POST['layout_popup_action_button_padding_left_tablet']),
                     'layout_popup_action_button_padding_bottom_tablet' => sanitize_text_field($_POST['layout_popup_action_button_padding_bottom_tablet']),
                     'layout_popup_action_button_padding_right_tablet'  => sanitize_text_field($_POST['layout_popup_action_button_padding_right_tablet']),
                     'layout_popup_action_button_padding_top_tablet'    => sanitize_text_field($_POST['layout_popup_action_button_padding_top_tablet']),
                     'layout_popup_action_button_padding_left_mobile'   => sanitize_text_field($_POST['layout_popup_action_button_padding_left_mobile']),
                     'layout_popup_action_button_padding_bottom_mobile' => sanitize_text_field($_POST['layout_popup_action_button_padding_bottom_mobile']),
                     'layout_popup_action_button_padding_right_mobile'  => sanitize_text_field($_POST['layout_popup_action_button_padding_right_mobile']),
                     'layout_popup_action_button_padding_top_mobile'    => sanitize_text_field($_POST['layout_popup_action_button_padding_top_mobile']),

                     'style_form_icon_color'                   => sanitize_text_field($_POST['style_form_icon_color']),
                     'style_form_icon_size'                    => sanitize_text_field($_POST['style_form_icon_size']),
                     'style_form_icon_size_tablet'             => sanitize_text_field($_POST['style_form_icon_size_tablet']),
                     'style_form_icon_size_mobile'             => sanitize_text_field($_POST['style_form_icon_size_mobile']),
                     'style_form_icon_position'                => sanitize_text_field($_POST['style_form_icon_position']),
                     'style_form_icon_padding_left'            => sanitize_text_field($_POST['style_form_icon_padding_left']),
                     'style_form_icon_padding_bottom'          => sanitize_text_field($_POST['style_form_icon_padding_bottom']),
                     'style_form_icon_padding_right'           => sanitize_text_field($_POST['style_form_icon_padding_right']),
                     'style_form_icon_padding_top'             => sanitize_text_field($_POST['style_form_icon_padding_top']),
                     'style_form_icon_padding_left_tablet'     => sanitize_text_field($_POST['style_form_icon_padding_left_tablet']),
                     'style_form_icon_padding_bottom_tablet'   => sanitize_text_field($_POST['style_form_icon_padding_bottom_tablet']),
                     'style_form_icon_padding_right_tablet'    => sanitize_text_field($_POST['style_form_icon_padding_right_tablet']),
                     'style_form_icon_padding_top_tablet'      => sanitize_text_field($_POST['style_form_icon_padding_top_tablet']),
                     'style_form_icon_padding_left_mobile'     => sanitize_text_field($_POST['style_form_icon_padding_left_mobile']),
                     'style_form_icon_padding_bottom_mobile'   => sanitize_text_field($_POST['style_form_icon_padding_bottom_mobile']),
                     'style_form_icon_padding_right_mobile'    => sanitize_text_field($_POST['style_form_icon_padding_right_mobile']),
                     'style_form_icon_padding_top_mobile'      => sanitize_text_field($_POST['style_form_icon_padding_top_mobile']),
                     'style_form_text_color'                   => sanitize_text_field($_POST['style_form_text_color']),
                     'style_form_font_size'                    => sanitize_text_field($_POST['style_form_font_size']),
                     'style_form_font_size_tablet'             => sanitize_text_field($_POST['style_form_font_size_tablet']),
                     'style_form_font_size_mobile'             => sanitize_text_field($_POST['style_form_font_size_mobile']),
                     'style_form_background_color'             => sanitize_text_field($_POST['style_form_background_color']),
                     'style_form_border_color'                 => sanitize_text_field($_POST['style_form_border_color']),
                     'style_form_border_width_left'            => sanitize_text_field($_POST['style_form_border_width_left']),
                     'style_form_border_width_bottom'          => sanitize_text_field($_POST['style_form_border_width_bottom']),
                     'style_form_border_width_right'           => sanitize_text_field($_POST['style_form_border_width_right']),
                     'style_form_border_width_top'             => sanitize_text_field($_POST['style_form_border_width_top']),
                     'style_form_border_width_left_tablet'     => sanitize_text_field($_POST['style_form_border_width_left_tablet']),
                     'style_form_border_width_bottom_tablet'   => sanitize_text_field($_POST['style_form_border_width_bottom_tablet']),
                     'style_form_border_width_right_tablet'    => sanitize_text_field($_POST['style_form_border_width_right_tablet']),
                     'style_form_border_width_top_tablet'      => sanitize_text_field($_POST['style_form_border_width_top_tablet']),
                     'style_form_border_width_left_mobile'     => sanitize_text_field($_POST['style_form_border_width_left_mobile']),
                     'style_form_border_width_bottom_mobile'   => sanitize_text_field($_POST['style_form_border_width_bottom_mobile']),
                     'style_form_border_width_right_mobile'    => sanitize_text_field($_POST['style_form_border_width_right_mobile']),
                     'style_form_border_width_top_mobile'      => sanitize_text_field($_POST['style_form_border_width_top_mobile']),
                     'style_form_border_radius_top_right'            => sanitize_text_field($_POST['style_form_border_radius_top_right']),
                     'style_form_border_radius_top_left'             => sanitize_text_field($_POST['style_form_border_radius_top_left']),
                     'style_form_border_radius_bottom_right'         => sanitize_text_field($_POST['style_form_border_radius_bottom_right']),
                     'style_form_border_radius_bottom_left'          => sanitize_text_field($_POST['style_form_border_radius_bottom_left']),
                     'style_form_border_radius_top_right_tablet'     => sanitize_text_field($_POST['style_form_border_radius_top_right_tablet']),
                     'style_form_border_radius_top_left_tablet'      => sanitize_text_field($_POST['style_form_border_radius_top_left_tablet']),
                     'style_form_border_radius_bottom_right_tablet'  => sanitize_text_field($_POST['style_form_border_radius_bottom_right_tablet']),
                     'style_form_border_radius_bottom_left_tablet'   => sanitize_text_field($_POST['style_form_border_radius_bottom_left_tablet']),
                     'style_form_border_radius_top_right_mobile'     => sanitize_text_field($_POST['style_form_border_radius_top_right_mobile']),
                     'style_form_border_radius_top_left_mobile'      => sanitize_text_field($_POST['style_form_border_radius_top_left_mobile']),
                     'style_form_border_radius_bottom_right_mobile'  => sanitize_text_field($_POST['style_form_border_radius_bottom_right_mobile']),
                     'style_form_border_radius_bottom_left_mobile'   => sanitize_text_field($_POST['style_form_border_radius_bottom_left_mobile']),

                     'style_form_padding_left'                 => sanitize_text_field($_POST['style_form_padding_left']),
                     'style_form_padding_bottom'               => sanitize_text_field($_POST['style_form_padding_bottom']),
                     'style_form_padding_right'                => sanitize_text_field($_POST['style_form_padding_right']),
                     'style_form_padding_top'                  => sanitize_text_field($_POST['style_form_padding_top']),
                     'style_form_padding_left_tablet'          => sanitize_text_field($_POST['style_form_padding_left_tablet']),
                     'style_form_padding_bottom_tablet'        => sanitize_text_field($_POST['style_form_padding_bottom_tablet']),
                     'style_form_padding_right_tablet'         => sanitize_text_field($_POST['style_form_padding_right_tablet']),
                     'style_form_padding_top_tablet'           => sanitize_text_field($_POST['style_form_padding_top_tablet']),
                     'style_form_padding_left_mobile'          => sanitize_text_field($_POST['style_form_padding_left_mobile']),
                     'style_form_padding_bottom_mobile'        => sanitize_text_field($_POST['style_form_padding_bottom_mobile']),
                     'style_form_padding_right_mobile'         => sanitize_text_field($_POST['style_form_padding_right_mobile']),
                     'style_form_padding_top_mobile'           => sanitize_text_field($_POST['style_form_padding_top_mobile']),

                     'style_search_form_post_type_select_text_color'                   => sanitize_text_field($_POST['style_search_form_post_type_select_text_color']),
                     'style_search_form_post_type_select_font_size'                    => sanitize_text_field($_POST['style_search_form_post_type_select_font_size']),
                     'style_search_form_post_type_select_font_size_tablet'             => sanitize_text_field($_POST['style_search_form_post_type_select_font_size_tablet']),
                     'style_search_form_post_type_select_font_size_mobile'             => sanitize_text_field($_POST['style_form_font_size_mobile']),
                     'style_search_form_post_type_select_background_color'             => sanitize_text_field($_POST['style_search_form_post_type_select_background_color']),
                     'style_search_form_post_type_select_border_color'                 => sanitize_text_field($_POST['style_search_form_post_type_select_border_color']),
                     'style_search_form_post_type_select_border_width_left'            => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_left']),
                     'style_search_form_post_type_select_border_width_bottom'          => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_bottom']),
                     'style_search_form_post_type_select_border_width_right'           => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_right']),
                     'style_search_form_post_type_select_border_width_top'             => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_top']),
                     'style_search_form_post_type_select_border_width_left_tablet'     => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_left_tablet']),
                     'style_search_form_post_type_select_border_width_bottom_tablet'   => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_bottom_tablet']),
                     'style_search_form_post_type_select_border_width_right_tablet'    => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_right_tablet']),
                     'style_search_form_post_type_select_border_width_top_tablet'      => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_top_tablet']),
                     'style_search_form_post_type_select_border_width_left_mobile'     => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_left_mobile']),
                     'style_search_form_post_type_select_border_width_bottom_mobile'   => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_bottom_mobile']),
                     'style_search_form_post_type_select_border_width_right_mobile'    => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_right_mobile']),
                     'style_search_form_post_type_select_border_width_top_mobile'      => sanitize_text_field($_POST['style_search_form_post_type_select_border_width_top_mobile']),
                     'style_search_form_post_type_select_border_radius_top_right'            => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_top_right']),
                     'style_search_form_post_type_select_border_radius_top_left'             => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_top_left']),
                     'style_search_form_post_type_select_border_radius_bottom_right'         => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_bottom_right']),
                     'style_search_form_post_type_select_border_radius_bottom_left'          => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_bottom_left']),
                     'style_search_form_post_type_select_border_radius_top_right_tablet'     => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_top_right_tablet']),
                     'style_search_form_post_type_select_border_radius_top_left_tablet'      => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_top_left_tablet']),
                     'style_search_form_post_type_select_border_radius_bottom_right_tablet'  => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_bottom_right_tablet']),
                     'style_search_form_post_type_select_border_radius_bottom_left_tablet'   => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_bottom_left_tablet']),
                     'style_search_form_post_type_select_border_radius_top_right_mobile'     => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_top_right_mobile']),
                     'style_search_form_post_type_select_border_radius_top_left_mobile'      => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_top_left_mobile']),
                     'style_search_form_post_type_select_border_radius_bottom_right_mobile'  => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_bottom_right_mobile']),
                     'style_search_form_post_type_select_border_radius_bottom_left_mobile'   => sanitize_text_field($_POST['style_search_form_post_type_select_border_radius_bottom_left_mobile']),

                     'style_search_form_post_type_select_padding_left'                 => sanitize_text_field($_POST['style_search_form_post_type_select_padding_left']),
                     'style_search_form_post_type_select_padding_bottom'               => sanitize_text_field($_POST['style_search_form_post_type_select_padding_bottom']),
                     'style_search_form_post_type_select_padding_right'                => sanitize_text_field($_POST['style_search_form_post_type_select_padding_right']),
                     'style_search_form_post_type_select_padding_top'                  => sanitize_text_field($_POST['style_search_form_post_type_select_padding_top']),
                     'style_search_form_post_type_select_padding_left_tablet'          => sanitize_text_field($_POST['style_search_form_post_type_select_padding_left_tablet']),
                     'style_search_form_post_type_select_padding_bottom_tablet'        => sanitize_text_field($_POST['style_search_form_post_type_select_padding_bottom_tablet']),
                     'style_search_form_post_type_select_padding_right_tablet'         => sanitize_text_field($_POST['style_search_form_post_type_select_padding_right_tablet']),
                     'style_search_form_post_type_select_padding_top_tablet'           => sanitize_text_field($_POST['style_search_form_post_type_select_padding_top_tablet']),
                     'style_search_form_post_type_select_padding_left_mobile'          => sanitize_text_field($_POST['style_search_form_post_type_select_padding_left_mobile']),
                     'style_search_form_post_type_select_padding_bottom_mobile'        => sanitize_text_field($_POST['style_search_form_post_type_select_padding_bottom_mobile']),
                     'style_search_form_post_type_select_padding_right_mobile'         => sanitize_text_field($_POST['style_search_form_post_type_select_padding_right_mobile']),
                     'style_search_form_post_type_select_padding_top_mobile'           => sanitize_text_field($_POST['style_search_form_post_type_select_padding_top_mobile']),
                     'style_search_form_post_type_select_margin_left'                 => sanitize_text_field($_POST['style_search_form_post_type_select_margin_left']),
                     'style_search_form_post_type_select_margin_bottom'               => sanitize_text_field($_POST['style_search_form_post_type_select_margin_bottom']),
                     'style_search_form_post_type_select_margin_right'                => sanitize_text_field($_POST['style_search_form_post_type_select_margin_right']),
                     'style_search_form_post_type_select_margin_top'                  => sanitize_text_field($_POST['style_search_form_post_type_select_margin_top']),
                     'style_search_form_post_type_select_margin_left_tablet'          => sanitize_text_field($_POST['style_search_form_post_type_select_margin_left_tablet']),
                     'style_search_form_post_type_select_margin_bottom_tablet'        => sanitize_text_field($_POST['style_search_form_post_type_select_margin_bottom_tablet']),
                     'style_search_form_post_type_select_margin_right_tablet'         => sanitize_text_field($_POST['style_search_form_post_type_select_margin_right_tablet']),
                     'style_search_form_post_type_select_margin_top_tablet'           => sanitize_text_field($_POST['style_search_form_post_type_select_margin_top_tablet']),
                     'style_search_form_post_type_select_margin_left_mobile'          => sanitize_text_field($_POST['style_search_form_post_type_select_margin_left_mobile']),
                     'style_search_form_post_type_select_margin_bottom_mobile'        => sanitize_text_field($_POST['style_search_form_post_type_select_margin_bottom_mobile']),
                     'style_search_form_post_type_select_margin_right_mobile'         => sanitize_text_field($_POST['style_search_form_post_type_select_margin_right_mobile']),
                     'style_search_form_post_type_select_margin_top_mobile'           => sanitize_text_field($_POST['style_search_form_post_type_select_margin_top_mobile']),

                     'style_search_form_post_type_select_hide_on_tablet' => isset($_POST['style_search_form_post_type_select_hide_on_tablet']) ? 'none' : 'block',
                     'style_search_form_post_type_select_hide_on_mobile' => isset($_POST['style_search_form_post_type_select_hide_on_mobile']) ? 'none' : 'block',

                     // Style: Search result
                     'style_result_box_max_height'                   => sanitize_text_field($_POST['style_result_box_max_height']),
                     'style_result_box_max_height_tablet'            => sanitize_text_field($_POST['style_result_box_max_height_tablet']),
                     'style_result_box_max_height_mobile'            => sanitize_text_field($_POST['style_result_box_max_height_mobile']),
                     'style_result_box_background_color'             => sanitize_text_field($_POST['style_result_box_background_color']),
                     'style_result_box_border_color'                 => sanitize_text_field($_POST['style_result_box_border_color']),
                     'style_result_box_border_width_left'            => sanitize_text_field($_POST['style_result_box_border_width_left']),
                     'style_result_box_border_width_bottom'          => sanitize_text_field($_POST['style_result_box_border_width_bottom']),
                     'style_result_box_border_width_right'           => sanitize_text_field($_POST['style_result_box_border_width_right']),
                     'style_result_box_border_width_top'             => sanitize_text_field($_POST['style_result_box_border_width_top']),
                     'style_result_box_border_width_left_tablet'     => sanitize_text_field($_POST['style_result_box_border_width_left_tablet']),
                     'style_result_box_border_width_bottom_tablet'   => sanitize_text_field($_POST['style_result_box_border_width_bottom_tablet']),
                     'style_result_box_border_width_right_tablet'    => sanitize_text_field($_POST['style_result_box_border_width_right_tablet']),
                     'style_result_box_border_width_top_tablet'      => sanitize_text_field($_POST['style_result_box_border_width_top_tablet']),
                     'style_result_box_border_width_left_mobile'     => sanitize_text_field($_POST['style_result_box_border_width_left_mobile']),
                     'style_result_box_border_width_bottom_mobile'   => sanitize_text_field($_POST['style_result_box_border_width_bottom_mobile']),
                     'style_result_box_border_width_right_mobile'    => sanitize_text_field($_POST['style_result_box_border_width_right_mobile']),
                     'style_result_box_border_width_top_mobile'      => sanitize_text_field($_POST['style_result_box_border_width_top_mobile']),

                     'style_result_box_border_radius_top_right'            => sanitize_text_field($_POST['style_result_box_border_radius_top_right']),
                     'style_result_box_border_radius_top_left'             => sanitize_text_field($_POST['style_result_box_border_radius_top_left']),
                     'style_result_box_border_radius_bottom_right'         => sanitize_text_field($_POST['style_result_box_border_radius_bottom_right']),
                     'style_result_box_border_radius_bottom_left'          => sanitize_text_field($_POST['style_result_box_border_radius_bottom_left']),
                     'style_result_box_border_radius_top_right_tablet'     => sanitize_text_field($_POST['style_result_box_border_radius_top_right_tablet']),
                     'style_result_box_border_radius_top_left_tablet'      => sanitize_text_field($_POST['style_result_box_border_radius_top_left_tablet']),
                     'style_result_box_border_radius_bottom_right_tablet'  => sanitize_text_field($_POST['style_result_box_border_radius_bottom_right_tablet']),
                     'style_result_box_border_radius_bottom_left_tablet'   => sanitize_text_field($_POST['style_result_box_border_radius_bottom_left_tablet']),
                     'style_result_box_border_radius_top_right_mobile'     => sanitize_text_field($_POST['style_result_box_border_radius_top_right_mobile']),
                     'style_result_box_border_radius_top_left_mobile'      => sanitize_text_field($_POST['style_result_box_border_radius_top_left_mobile']),
                     'style_result_box_border_radius_bottom_right_mobile'  => sanitize_text_field($_POST['style_result_box_border_radius_bottom_right_mobile']),
                     'style_result_box_border_radius_bottom_left_mobile'   => sanitize_text_field($_POST['style_result_box_border_radius_bottom_left_mobile']),

                     'style_result_box_e_padding_top'                   => sanitize_text_field($_POST['style_result_box_e_padding_top']),
                     'style_result_box_e_padding_top_tablet'            => sanitize_text_field($_POST['style_result_box_e_padding_top_tablet']),
                     'style_result_box_e_padding_top_mobile'            => sanitize_text_field($_POST['style_result_box_e_padding_top_mobile']),

                     'style_result_box_padding_left'                 => sanitize_text_field($_POST['style_result_box_padding_left']),
                     'style_result_box_padding_bottom'               => sanitize_text_field($_POST['style_result_box_padding_bottom']),
                     'style_result_box_padding_right'                => sanitize_text_field($_POST['style_result_box_padding_right']),
                     'style_result_box_padding_top'                  => sanitize_text_field($_POST['style_result_box_padding_top']),
                     'style_result_box_padding_left_tablet'          => sanitize_text_field($_POST['style_result_box_padding_left_tablet']),
                     'style_result_box_padding_bottom_tablet'        => sanitize_text_field($_POST['style_result_box_padding_bottom_tablet']),
                     'style_result_box_padding_right_tablet'         => sanitize_text_field($_POST['style_result_box_padding_right_tablet']),
                     'style_result_box_padding_top_tablet'           => sanitize_text_field($_POST['style_result_box_padding_top_tablet']),
                     'style_result_box_padding_left_mobile'          => sanitize_text_field($_POST['style_result_box_padding_left_mobile']),
                     'style_result_box_padding_bottom_mobile'        => sanitize_text_field($_POST['style_result_box_padding_bottom_mobile']),
                     'style_result_box_padding_right_mobile'         => sanitize_text_field($_POST['style_result_box_padding_right_mobile']),
                     'style_result_box_padding_top_mobile'           => sanitize_text_field($_POST['style_result_box_padding_top_mobile']),
                     'style_result_box_margin_left'                  => sanitize_text_field($_POST['style_result_box_margin_left']),
                     'style_result_box_margin_bottom'                => sanitize_text_field($_POST['style_result_box_margin_bottom']),
                     'style_result_box_margin_right'                 => sanitize_text_field($_POST['style_result_box_margin_right']),
                     'style_result_box_margin_top'                   => sanitize_text_field($_POST['style_result_box_margin_top']),
                     'style_result_box_margin_left_tablet'           => sanitize_text_field($_POST['style_result_box_margin_left_tablet']),
                     'style_result_box_margin_bottom_tablet'         => sanitize_text_field($_POST['style_result_box_margin_bottom_tablet']),
                     'style_result_box_margin_right_tablet'          => sanitize_text_field($_POST['style_result_box_margin_right_tablet']),
                     'style_result_box_margin_top_tablet'            => sanitize_text_field($_POST['style_result_box_margin_top_tablet']),
                     'style_result_box_margin_left_mobile'           => sanitize_text_field($_POST['style_result_box_margin_left_mobile']),
                     'style_result_box_margin_bottom_mobile'         => sanitize_text_field($_POST['style_result_box_margin_bottom_mobile']),
                     'style_result_box_margin_right_mobile'          => sanitize_text_field($_POST['style_result_box_margin_right_mobile']),
                     'style_result_box_margin_top_mobile'            => sanitize_text_field($_POST['style_result_box_margin_top_mobile']),

                     'style_search_result_item_icon_color'              => sanitize_text_field($_POST['style_search_result_item_icon_color']),
                     'style_search_result_item_icon_size'               => sanitize_text_field($_POST['style_search_result_item_icon_size']),
                     'style_search_result_item_icon_size_tablet'        => sanitize_text_field($_POST['style_search_result_item_icon_size_tablet']),
                     'style_search_result_item_icon_size_mobile'        => sanitize_text_field($_POST['style_search_result_item_icon_size_mobile']),
                     'style_search_result_item_link_color'              => sanitize_text_field($_POST['style_search_result_item_link_color']),
                     'style_search_result_item_link_hover_color'        => sanitize_text_field($_POST['style_search_result_item_link_hover_color']),
                     'post_type_product_results_price_color'            => sanitize_text_field($_POST['post_type_product_results_price_color']),
                     'style_search_result_item_link_font_size'          => sanitize_text_field($_POST['style_search_result_item_link_font_size']),
                     'style_search_result_item_link_font_size_tablet'   => sanitize_text_field($_POST['style_search_result_item_link_font_size_tablet']),
                     'style_search_result_item_link_font_size_mobile'   => sanitize_text_field($_POST['style_search_result_item_link_font_size_mobile']),
                     'post_type_product_results_price_font_size'            => sanitize_text_field($_POST['post_type_product_results_price_font_size']),
                     'post_type_product_results_price_font_size_tablet'     => sanitize_text_field($_POST['post_type_product_results_price_font_size_tablet']),
                     'post_type_product_results_price_font_size_mobile'     => sanitize_text_field($_POST['post_type_product_results_price_font_size_mobile']),
                     'style_search_result_item_spacing'                 => sanitize_text_field($_POST['style_search_result_item_spacing']),
                     'style_search_result_item_spacing_tablet'          => sanitize_text_field($_POST['style_search_result_item_spacing_tablet']),
                     'style_search_result_item_spacing_mobile'          => sanitize_text_field($_POST['style_search_result_item_spacing_mobile']),

                     'style_related_result_icon_color'                  => sanitize_text_field($_POST['style_related_result_icon_color']),
                     'style_related_result_icon_size'                   => sanitize_text_field($_POST['style_related_result_icon_size']),
                     'style_related_result_icon_size_tablet'            => sanitize_text_field($_POST['style_related_result_icon_size_tablet']),
                     'style_related_result_icon_size_mobile'            => sanitize_text_field($_POST['style_related_result_icon_size_mobile']),
                     'style_related_result_link_color'                  => sanitize_text_field($_POST['style_related_result_link_color']),
                     'style_related_result_link_hover_color'            => sanitize_text_field($_POST['style_related_result_link_hover_color']),
                     'style_related_result_link_font_size'              => sanitize_text_field($_POST['style_related_result_link_font_size']),
                     'style_related_result_link_font_size_tablet'       => sanitize_text_field($_POST['style_related_result_link_font_size_tablet']),
                     'style_related_result_link_font_size_mobile'       => sanitize_text_field($_POST['style_related_result_link_font_size_mobile']),
                     'style_related_result_item_spacing'                => sanitize_text_field($_POST['style_related_result_item_spacing']),
                     'style_related_result_item_spacing_tablet'         => sanitize_text_field($_POST['style_related_result_item_spacing_tablet']),
                     'style_related_result_item_spacing_mobile'         => sanitize_text_field($_POST['style_related_result_item_spacing_mobile']),
                     'style_related_result_border_color'                => sanitize_text_field($_POST['style_related_result_border_color']),
                     'style_related_result_border_width_left'           => sanitize_text_field($_POST['style_related_result_border_width_left']),
                     'style_related_result_border_width_bottom'         => sanitize_text_field($_POST['style_related_result_border_width_bottom']),
                     'style_related_result_border_width_right'          => sanitize_text_field($_POST['style_related_result_border_width_right']),
                     'style_related_result_border_width_top'            => sanitize_text_field($_POST['style_related_result_border_width_top']),
                     'style_related_result_border_width_left_tablet'    => sanitize_text_field($_POST['style_related_result_border_width_left_tablet']),
                     'style_related_result_border_width_bottom_tablet'  => sanitize_text_field($_POST['style_related_result_border_width_bottom_tablet']),
                     'style_related_result_border_width_right_tablet'   => sanitize_text_field($_POST['style_related_result_border_width_right_tablet']),
                     'style_related_result_border_width_top_tablet'     => sanitize_text_field($_POST['style_related_result_border_width_top_tablet']),
                     'style_related_result_border_width_left_mobile'    => sanitize_text_field($_POST['style_related_result_border_width_left_mobile']),
                     'style_related_result_border_width_bottom_mobile'  => sanitize_text_field($_POST['style_related_result_border_width_bottom_mobile']),
                     'style_related_result_border_width_right_mobile'   => sanitize_text_field($_POST['style_related_result_border_width_right_mobile']),
                     'style_related_result_border_width_top_mobile'     => sanitize_text_field($_POST['style_related_result_border_width_top_mobile']),
                  ];

                  if ($form_data->form_options['search_result_carousel'] == 1) {
                     $arguments_search_result_carousel = [
                        'style_sr_ci_background'                  => sanitize_text_field($_POST['style_sr_ci_background']),
                        'style_sr_ci_border_color'                => sanitize_text_field($_POST['style_sr_ci_border_color']),
                        'style_sr_ci_border_width_left'           => sanitize_text_field($_POST['style_sr_ci_border_width_left']),
                        'style_sr_ci_border_width_bottom'         => sanitize_text_field($_POST['style_sr_ci_border_width_bottom']),
                        'style_sr_ci_border_width_right'          => sanitize_text_field($_POST['style_sr_ci_border_width_right']),
                        'style_sr_ci_border_width_top'            => sanitize_text_field($_POST['style_sr_ci_border_width_top']),
                        'style_sr_ci_border_width_left_tablet'    => sanitize_text_field($_POST['style_sr_ci_border_width_left_tablet']),
                        'style_sr_ci_border_width_bottom_tablet'  => sanitize_text_field($_POST['style_sr_ci_border_width_bottom_tablet']),
                        'style_sr_ci_border_width_right_tablet'   => sanitize_text_field($_POST['style_sr_ci_border_width_right_tablet']),
                        'style_sr_ci_border_width_top_tablet'     => sanitize_text_field($_POST['style_sr_ci_border_width_top_tablet']),
                        'style_sr_ci_border_width_left_mobile'    => sanitize_text_field($_POST['style_sr_ci_border_width_left_mobile']),
                        'style_sr_ci_border_width_bottom_mobile'  => sanitize_text_field($_POST['style_sr_ci_border_width_bottom_mobile']),
                        'style_sr_ci_border_width_right_mobile'   => sanitize_text_field($_POST['style_sr_ci_border_width_right_mobile']),
                        'style_sr_ci_border_width_top_mobile'     => sanitize_text_field($_POST['style_sr_ci_border_width_top_mobile']),
                        'style_sr_ci_border_radius'               => sanitize_text_field($_POST['style_sr_ci_border_radius']),
                        'style_sr_ci_border_radius_tablet'        => sanitize_text_field($_POST['style_sr_ci_border_radius_tablet']),
                        'style_sr_ci_border_radius_mobile'        => sanitize_text_field($_POST['style_sr_ci_border_radius_mobile']),
                        'style_sr_ci_thumbnail_height'            => sanitize_text_field($_POST['style_sr_ci_thumbnail_height']),
                        'style_sr_ci_thumbnail_height_tablet'     => sanitize_text_field($_POST['style_sr_ci_thumbnail_height_tablet']),
                        'style_sr_ci_thumbnail_height_mobile'     => sanitize_text_field($_POST['style_sr_ci_thumbnail_height_mobile']),
                        'style_sr_ci_padding_left'                => sanitize_text_field($_POST['style_sr_ci_padding_left']),
                        'style_sr_ci_padding_bottom'              => sanitize_text_field($_POST['style_sr_ci_padding_bottom']),
                        'style_sr_ci_padding_right'               => sanitize_text_field($_POST['style_sr_ci_padding_right']),
                        'style_sr_ci_padding_top'                 => sanitize_text_field($_POST['style_sr_ci_padding_top']),
                        'style_sr_ci_padding_left_tablet'         => sanitize_text_field($_POST['style_sr_ci_padding_left_tablet']),
                        'style_sr_ci_padding_bottom_tablet'       => sanitize_text_field($_POST['style_sr_ci_padding_bottom_tablet']),
                        'style_sr_ci_padding_right_tablet'        => sanitize_text_field($_POST['style_sr_ci_padding_right_tablet']),
                        'style_sr_ci_padding_top_tablet'          => sanitize_text_field($_POST['style_sr_ci_padding_top_tablet']),
                        'style_sr_ci_padding_left_mobile'         => sanitize_text_field($_POST['style_sr_ci_padding_left_mobile']),
                        'style_sr_ci_padding_bottom_mobile'       => sanitize_text_field($_POST['style_sr_ci_padding_bottom_mobile']),
                        'style_sr_ci_padding_right_mobile'        => sanitize_text_field($_POST['style_sr_ci_padding_right_mobile']),
                        'style_sr_ci_padding_top_mobile'          => sanitize_text_field($_POST['style_sr_ci_padding_top_mobile']),
                     ];

                     $arguments = array_merge($arguments, $arguments_search_result_carousel);
                  }

                  if ($form_data->form_options['search_banner_enable'] == 1) {
                     $arguments_search_banner = [
                        'style_banner_border_radius'           => sanitize_text_field($_POST['style_banner_border_radius']),
                        'style_banner_border_radius_tablet'    => sanitize_text_field($_POST['style_banner_border_radius_tablet']),
                        'style_banner_border_radius_mobile'    => sanitize_text_field($_POST['style_banner_border_radius_mobile']),
                        'style_banner_alignment'               => sanitize_text_field($_POST['style_banner_alignment']),
                        'style_banner_width'                   => sanitize_text_field($_POST['style_banner_width']),
                        'style_banner_width_tablet'            => sanitize_text_field($_POST['style_banner_width_tablet']),
                        'style_banner_width_mobile'            => sanitize_text_field($_POST['style_banner_width_mobile']),
                        'style_banner_height'                  => sanitize_text_field($_POST['style_banner_height']),
                        'style_banner_height_tablet'           => sanitize_text_field($_POST['style_banner_height_tablet']),
                        'style_banner_height_mobile'           => sanitize_text_field($_POST['style_banner_height_mobile']),
                        'style_banner_padding_left'            => sanitize_text_field($_POST['style_banner_padding_left']),
                        'style_banner_padding_bottom'          => sanitize_text_field($_POST['style_banner_padding_bottom']),
                        'style_banner_padding_right'           => sanitize_text_field($_POST['style_banner_padding_right']),
                        'style_banner_padding_top'             => sanitize_text_field($_POST['style_banner_padding_top']),
                        'style_banner_padding_left_tablet'     => sanitize_text_field($_POST['style_banner_padding_left_tablet']),
                        'style_banner_padding_bottom_tablet'   => sanitize_text_field($_POST['style_banner_padding_bottom_tablet']),
                        'style_banner_padding_right_tablet'    => sanitize_text_field($_POST['style_banner_padding_right_tablet']),
                        'style_banner_padding_top_tablet'      => sanitize_text_field($_POST['style_banner_padding_top_tablet']),
                        'style_banner_padding_left_mobile'     => sanitize_text_field($_POST['style_banner_padding_left_mobile']),
                        'style_banner_padding_bottom_mobile'   => sanitize_text_field($_POST['style_banner_padding_bottom_mobile']),
                        'style_banner_padding_right_mobile'    => sanitize_text_field($_POST['style_banner_padding_right_mobile']),
                        'style_banner_padding_top_mobile'      => sanitize_text_field($_POST['style_banner_padding_top_mobile']),
                     ];

                     $arguments = array_merge($arguments, $arguments_search_banner);
                  }

                  if ($form_data->form_options['user_last_searches_enable'] == 1) {
                     $arguments_user_last_searches = [
                        'style_user_last_searches_icon_color'                    => sanitize_text_field($_POST['style_user_last_searches_icon_color']),
                        'style_user_last_searches_icon_size'                     => sanitize_text_field($_POST['style_user_last_searches_icon_size']),
                        'style_user_last_searches_icon_size_tablet'              => sanitize_text_field($_POST['style_user_last_searches_icon_size_tablet']),
                        'style_user_last_searches_icon_size_mobile'              => sanitize_text_field($_POST['style_user_last_searches_icon_size_mobile']),
                        'style_user_last_searches_heading_color'                 => sanitize_text_field($_POST['style_user_last_searches_heading_color']),
                        'style_user_last_searches_heading_font_size'             => sanitize_text_field($_POST['style_user_last_searches_heading_font_size']),
                        'style_user_last_searches_heading_font_size_tablet'      => sanitize_text_field($_POST['style_user_last_searches_heading_font_size_tablet']),
                        'style_user_last_searches_heading_font_size_mobile'      => sanitize_text_field($_POST['style_user_last_searches_heading_font_size_mobile']),
                        'style_user_last_searches_heading_margin_bottom'             => sanitize_text_field($_POST['style_user_last_searches_heading_margin_bottom']),
                        'style_user_last_searches_heading_margin_bottom_tablet'      => sanitize_text_field($_POST['style_user_last_searches_heading_margin_bottom_tablet']),
                        'style_user_last_searches_heading_margin_bottom_mobile'      => sanitize_text_field($_POST['style_user_last_searches_heading_margin_bottom_mobile']),
                        'style_user_last_searches_item_link_color'               => sanitize_text_field($_POST['style_user_last_searches_item_link_color']),
                        'style_user_last_searches_item_link_hover_color'         => sanitize_text_field($_POST['style_user_last_searches_item_link_hover_color']),
                        'style_user_last_searches_item_link_arrow_color'         => sanitize_text_field($_POST['style_user_last_searches_item_link_arrow_color']),
                        'style_user_last_searches_item_link_arrow_hover_color'   => sanitize_text_field($_POST['style_user_last_searches_item_link_arrow_hover_color']),
                        'style_user_last_searches_item_link_font_size'           => sanitize_text_field($_POST['style_user_last_searches_item_link_font_size']),
                        'style_user_last_searches_item_link_font_size_tablet'    => sanitize_text_field($_POST['style_user_last_searches_item_link_font_size_tablet']),
                        'style_user_last_searches_item_link_font_size_mobile'    => sanitize_text_field($_POST['style_user_last_searches_item_link_font_size_mobile']),
                        'style_user_last_searches_item_border_radius'            => sanitize_text_field($_POST['style_user_last_searches_item_border_radius']),
                        'style_user_last_searches_item_border_radius_tablet'     => sanitize_text_field($_POST['style_user_last_searches_item_border_radius_tablet']),
                        'style_user_last_searches_item_border_radius_mobile'     => sanitize_text_field($_POST['style_user_last_searches_item_border_radius_mobile']),
                        'style_user_last_searches_padding_bottom'                => sanitize_text_field($_POST['style_user_last_searches_padding_bottom']),
                        'style_user_last_searches_padding_right'                 => sanitize_text_field($_POST['style_user_last_searches_padding_right']),
                        'style_user_last_searches_padding_top'                   => sanitize_text_field($_POST['style_user_last_searches_padding_top']),
                        'style_user_last_searches_padding_left_tablet'           => sanitize_text_field($_POST['style_user_last_searches_padding_left_tablet']),
                        'style_user_last_searches_padding_bottom_tablet'         => sanitize_text_field($_POST['style_user_last_searches_padding_bottom_tablet']),
                        'style_user_last_searches_padding_right_tablet'          => sanitize_text_field($_POST['style_user_last_searches_padding_right_tablet']),
                        'style_user_last_searches_padding_top_tablet'            => sanitize_text_field($_POST['style_user_last_searches_padding_top_tablet']),
                        'style_user_last_searches_padding_left_mobile'           => sanitize_text_field($_POST['style_user_last_searches_padding_left_mobile']),
                        'style_user_last_searches_padding_bottom_mobile'         => sanitize_text_field($_POST['style_user_last_searches_padding_bottom_mobile']),
                        'style_user_last_searches_padding_right_mobile'          => sanitize_text_field($_POST['style_user_last_searches_padding_right_mobile']),
                        'style_user_last_searches_padding_top_mobile'            => sanitize_text_field($_POST['style_user_last_searches_padding_top_mobile']),
                     ];

                     $arguments = array_merge($arguments, $arguments_user_last_searches);
                  }

                  if ($form_data->form_options['papular_searches_enable'] == 1) {
                     $arguments_papular_searches = [
                        'style_papular_searches_icon_color'                   => sanitize_text_field($_POST['style_papular_searches_icon_color']),
                        'style_papular_searches_icon_size'                    => sanitize_text_field($_POST['style_papular_searches_icon_size']),
                        'style_papular_searches_icon_size_tablet'             => sanitize_text_field($_POST['style_papular_searches_icon_size_tablet']),
                        'style_papular_searches_icon_size_mobile'             => sanitize_text_field($_POST['style_papular_searches_icon_size_mobile']),
                        'style_papular_searches_heading_color'                => sanitize_text_field($_POST['style_papular_searches_heading_color']),
                        'style_papular_searches_heading_font_size'            => sanitize_text_field($_POST['style_papular_searches_heading_font_size']),
                        'style_papular_searches_heading_font_size_tablet'     => sanitize_text_field($_POST['style_papular_searches_heading_font_size_tablet']),
                        'style_papular_searches_heading_font_size_mobile'     => sanitize_text_field($_POST['style_papular_searches_heading_font_size_mobile']),
                        'style_papular_searches_heading_margin_bottom'        => sanitize_text_field($_POST['style_papular_searches_heading_margin_bottom']),
                        'style_papular_searches_heading_margin_bottom_tablet' => sanitize_text_field($_POST['style_papular_searches_heading_margin_bottom_tablet']),
                        'style_papular_searches_heading_margin_bottom_mobile' => sanitize_text_field($_POST['style_papular_searches_heading_margin_bottom_mobile']),
                        'style_papular_searches_item_link_color'              => sanitize_text_field($_POST['style_papular_searches_item_link_color']),
                        'style_papular_searches_item_link_hover_color'        => sanitize_text_field($_POST['style_papular_searches_item_link_hover_color']),
                        'style_papular_searches_item_link_arrow_color'        => sanitize_text_field($_POST['style_papular_searches_item_link_arrow_color']),
                        'style_papular_searches_item_link_arrow_hover_color'  => sanitize_text_field($_POST['style_papular_searches_item_link_arrow_hover_color']),
                        'style_papular_searches_item_link_font_size'          => sanitize_text_field($_POST['style_papular_searches_item_link_font_size']),
                        'style_papular_searches_item_link_font_size_tablet'   => sanitize_text_field($_POST['style_papular_searches_item_link_font_size_tablet']),
                        'style_papular_searches_item_link_font_size_mobile'   => sanitize_text_field($_POST['style_papular_searches_item_link_font_size_mobile']),
                        'style_papular_searches_item_border_radius'           => sanitize_text_field($_POST['style_papular_searches_item_border_radius']),
                        'style_papular_searches_item_border_radius_tablet'    => sanitize_text_field($_POST['style_papular_searches_item_border_radius_tablet']),
                        'style_papular_searches_item_border_radius_mobile'    => sanitize_text_field($_POST['style_papular_searches_item_border_radius_mobile']),
                        'style_papular_searches_padding_left'                 => sanitize_text_field($_POST['style_papular_searches_padding_left']),
                        'style_papular_searches_padding_bottom'               => sanitize_text_field($_POST['style_papular_searches_padding_bottom']),
                        'style_papular_searches_padding_right'                => sanitize_text_field($_POST['style_papular_searches_padding_right']),
                        'style_papular_searches_padding_top'                  => sanitize_text_field($_POST['style_papular_searches_padding_top']),
                        'style_papular_searches_padding_left_tablet'          => sanitize_text_field($_POST['style_papular_searches_padding_left_tablet']),
                        'style_papular_searches_padding_bottom_tablet'        => sanitize_text_field($_POST['style_papular_searches_padding_bottom_tablet']),
                        'style_papular_searches_padding_right_tablet'         => sanitize_text_field($_POST['style_papular_searches_padding_right_tablet']),
                        'style_papular_searches_padding_top_tablet'           => sanitize_text_field($_POST['style_papular_searches_padding_top_tablet']),
                        'style_papular_searches_padding_left_mobile'          => sanitize_text_field($_POST['style_papular_searches_padding_left_mobile']),
                        'style_papular_searches_padding_bottom_mobile'        => sanitize_text_field($_POST['style_papular_searches_padding_bottom_mobile']),
                        'style_papular_searches_padding_right_mobile'         => sanitize_text_field($_POST['style_papular_searches_padding_right_mobile']),
                        'style_papular_searches_padding_top_mobile'           => sanitize_text_field($_POST['style_papular_searches_padding_top_mobile']),
                     ];

                     $arguments = array_merge($arguments, $arguments_papular_searches);
                  }

                  if ($form_data->form_options['suggested_categories_enable'] == 1) {
                     $arguments_suggested_categories = [
                        'style_suggested_categories_icon_color'                  => sanitize_text_field($_POST['style_suggested_categories_icon_color']),
                        'style_suggested_categories_icon_size'                   => sanitize_text_field($_POST['style_suggested_categories_icon_size']),
                        'style_suggested_categories_icon_size_tablet'            => sanitize_text_field($_POST['style_suggested_categories_icon_size_tablet']),
                        'style_suggested_categories_icon_size_mobile'            => sanitize_text_field($_POST['style_suggested_categories_icon_size_mobile']),
                        'style_suggested_categories_heading_color'               => sanitize_text_field($_POST['style_suggested_categories_heading_color']),
                        'style_suggested_categories_heading_font_size'           => sanitize_text_field($_POST['style_suggested_categories_heading_font_size']),
                        'style_suggested_categories_heading_font_size_tablet'    => sanitize_text_field($_POST['style_suggested_categories_heading_font_size_tablet']),
                        'style_suggested_categories_heading_font_size_mobile'    => sanitize_text_field($_POST['style_suggested_categories_heading_font_size_mobile']),
                        'style_suggested_categories_heading_margin_bottom'          => sanitize_text_field($_POST['style_suggested_categories_heading_margin_bottom']),
                        'style_suggested_categories_heading_margin_bottom_tablet'   => sanitize_text_field($_POST['style_suggested_categories_heading_margin_bottom_tablet']),
                        'style_suggested_categories_heading_margin_bottom_mobile'   => sanitize_text_field($_POST['style_suggested_categories_heading_margin_bottom_mobile']),
                        'style_suggested_categories_item_link_color'             => sanitize_text_field($_POST['style_suggested_categories_item_link_color']),
                        'style_suggested_categories_item_link_hover_color'       => sanitize_text_field($_POST['style_suggested_categories_item_link_hover_color']),
                        'style_suggested_categories_item_link_arrow_color'       => sanitize_text_field($_POST['style_suggested_categories_item_link_arrow_color']),
                        'style_suggested_categories_item_link_arrow_hover_color' => sanitize_text_field($_POST['style_suggested_categories_item_link_arrow_hover_color']),
                        'style_suggested_categories_item_link_font_size'         => sanitize_text_field($_POST['style_suggested_categories_item_link_font_size']),
                        'style_suggested_categories_item_link_font_size_tablet'  => sanitize_text_field($_POST['style_suggested_categories_item_link_font_size_tablet']),
                        'style_suggested_categories_item_link_font_size_mobile'  => sanitize_text_field($_POST['style_suggested_categories_item_link_font_size_mobile']),
                        'style_suggested_categories_item_border_radius'          => sanitize_text_field($_POST['style_suggested_categories_item_border_radius']),
                        'style_suggested_categories_item_border_radius_tablet'   => sanitize_text_field($_POST['style_suggested_categories_item_border_radius_tablet']),
                        'style_suggested_categories_item_border_radius_mobile'   => sanitize_text_field($_POST['style_suggested_categories_item_border_radius_mobile']),
                        'style_suggested_categories_padding_left'                => sanitize_text_field($_POST['style_suggested_categories_padding_left']),
                        'style_suggested_categories_padding_bottom'              => sanitize_text_field($_POST['style_suggested_categories_padding_bottom']),
                        'style_suggested_categories_padding_right'               => sanitize_text_field($_POST['style_suggested_categories_padding_right']),
                        'style_suggested_categories_padding_top'                 => sanitize_text_field($_POST['style_suggested_categories_padding_top']),
                        'style_suggested_categories_padding_left_tablet'         => sanitize_text_field($_POST['style_suggested_categories_padding_left_tablet']),
                        'style_suggested_categories_padding_bottom_tablet'       => sanitize_text_field($_POST['style_suggested_categories_padding_bottom_tablet']),
                        'style_suggested_categories_padding_right_tablet'        => sanitize_text_field($_POST['style_suggested_categories_padding_right_tablet']),
                        'style_suggested_categories_padding_top_tablet'          => sanitize_text_field($_POST['style_suggested_categories_padding_top_tablet']),
                        'style_suggested_categories_padding_left_mobile'         => sanitize_text_field($_POST['style_suggested_categories_padding_left_mobile']),
                        'style_suggested_categories_padding_bottom_mobile'       => sanitize_text_field($_POST['style_suggested_categories_padding_bottom_mobile']),
                        'style_suggested_categories_padding_right_mobile'        => sanitize_text_field($_POST['style_suggested_categories_padding_right_mobile']),
                        'style_suggested_categories_padding_top_mobile'          => sanitize_text_field($_POST['style_suggested_categories_padding_top_mobile']),
                     ];

                     $arguments = array_merge($arguments, $arguments_suggested_categories);
                  }

                  $file       = wfs_PP . "/backups/form-settings-{$form_id}.json";
                  $saved_json = file_get_contents($file);
                  $saved_arr  = json_decode($saved_json, true);

                  foreach ($arguments as $opt_key => $opt) {
                     $saved_arr[$opt_key] = $opt;
                  }

                  file_put_contents($file, wp_json_encode($saved_arr));

                  WFS_DB_Queries::update_form_option((int) $form_id, $arguments);

                  wp_redirect($this->request_uri . '&success=1');
                  exit;
               }
            } elseif ($current_tab == 'translate') {
               $dropdown_language_selected = get_locale();
               $form_translates_values = [];

               if (isset($_GET['lang']) && !empty($_GET['lang'])) {
                  $dropdown_language_selected = (string) sanitize_text_field($_GET['lang']);
                  $lang = (string) sanitize_text_field($_GET['lang']);
                  $translate = 'translate_' . $lang;

                  if (isset($form_data->form_options[$translate]) && !empty($form_data->form_options[$translate])) {
                     $form_translates_values = $form_data->form_options[$translate];
                  }
               } else {
                  wp_redirect("{$this->request_uri}&lang={$dropdown_language_selected}");
                  exit;
               }

               if (isset($_POST['translate_change_lang_btn']) && isset($_POST['translate_change_lang_slug']) && !empty($_POST['translate_change_lang_slug'])) {
                  wp_redirect("{$this->request_uri}&lang={$_POST['translate_change_lang_slug']}");
                  exit;
               }

               if (isset($_POST['form_edit_submit'])) {
                  $translates = [
                     'input_placeholder'           => sanitize_text_field($_POST['input_placeholder']),
                     'user_last_searches_title'    => sanitize_text_field($_POST['user_last_searches_title']),
                     'papular_searches_title'      => sanitize_text_field($_POST['papular_searches_title']),
                     'suggested_categories_title'  => sanitize_text_field($_POST['suggested_categories_title']),
                     'no_result'                   => sanitize_text_field($_POST['no_result']),
                  ];

                  $save_lang = [
                     'translate_' . (string) sanitize_text_field($_GET['lang']) => $translates
                  ];

                  $file       = wfs_PP . "/backups/form-settings-{$form_id}.json";
                  $saved_json = file_get_contents($file);
                  $saved_arr  = json_decode($saved_json, true);

                  foreach ($save_lang as $opt_key => $opt) {
                     $saved_arr[$opt_key] = $opt;
                  }

                  file_put_contents($file, wp_json_encode($saved_arr));

                  WFS_DB_Queries::update_form_option((int) $form_id, $save_lang);

                  wp_redirect("{$this->request_uri}&lang={$_GET['lang']}&success=1");
                  exit;
               }
            }

            $template = wfs_PP . '/templates/admin/form-edit.php';
         }

         if (!file_exists($template)) {
            die;
         }

         require_once $template;
      }

      public function register_settings_callback()
      {
         $current_tab = isset($_GET['tab']) && in_array($_GET['tab'], ['general']) ? $_GET['tab'] : 'general';

         $this->request_uri = $_SERVER['REQUEST_URI'];

         if (isset($_POST['settings_submit'])) {
            if ($current_tab == 'general') {
               update_option('wfs_settings_calendar', $_POST['wfs_settings_calendar']);
            }

            wp_redirect($this->request_uri . '&success=1');
            exit;
         }

         $template = wfs_PP . '/templates/admin/settings.php';

         if (!file_exists($template)) {
            die;
         }

         require_once $template;
      }
   }
   WFS_Admin_Page::ins();
}
